/*
 *	filterCommon.cpp - Helper routines for filter classes
 *	Copyright (C) 2018, D Haley 

 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.

 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.

 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "filterCommon.h"

#include "common/colourmap.h"
#include "wx/wxcommon.h"


#include "shull/NA_wrapper/NewtonApple_hull3D.h"


//TODO: Work out where the payoff for this is

using std::ostream;
using std::vector;
using std::endl;
using std::string;

void writeVectorsXML(ostream &f,const char *containerName,
		const vector<Point3D> &vectorParams, unsigned int depth)
{
	f << tabs(depth+1) << "<" << containerName << ">" << endl;
	for(unsigned int ui=0; ui<vectorParams.size(); ui++)
	{
		f << tabs(depth+2) << "<point3d x=\"" << vectorParams[ui][0] << 
			"\" y=\"" << vectorParams[ui][1] << "\" z=\"" << vectorParams[ui][2] << "\"/>" << endl;
	}
	f << tabs(depth+1) << "</" << containerName << ">" << endl;
}

void writeIonsEnabledXML(ostream &f, const char *containerName, 
		const vector<bool> &enabledState, const vector<string> &names, 
			unsigned int depth)
{
	if(enabledState.size()!=names.size())
		return;

	f << tabs(depth) << "<" << containerName << ">"  << endl;
	for(size_t ui=0;ui<enabledState.size();ui++)
	{
		f<< tabs(depth+1) << "<ion enabled=\"" << (int)enabledState[ui] 
			<< "\" name=\"" << names[ui] << "\"/>" <<  std::endl; 
	}
	f << tabs(depth) << "</" << containerName << ">"  << endl;
}

void readIonsEnabledXML(xmlNodePtr nodePtr,  vector<bool> &enabledStatus,vector<string> &ionNames)
{
	//skip conatainer name
	nodePtr=nodePtr->xmlChildrenNode;

	if(!nodePtr)
		return;

	enabledStatus.clear();
	while(!XMLHelpFwdToElem(nodePtr,"ion"))
	{
		int enabled;
		if(!XMLGetAttrib(nodePtr,enabled,"enabled"))
			return ;

		std::string tmpName;
		if(!XMLGetAttrib(nodePtr,tmpName,"name"))
			return;
	
		enabledStatus.push_back(enabled);
		
		ionNames.push_back(tmpName);
	}
	
}
bool readVectorsXML(xmlNodePtr nodePtr,	std::vector<Point3D> &vectorParams) 
{
	nodePtr=nodePtr->xmlChildrenNode;
	vectorParams.clear();
	
	while(!XMLHelpFwdToElem(nodePtr,"point3d"))
	{
		std::string tmpStr;
		xmlChar* xmlString;
		float x,y,z;
		//--Get X value--
		xmlString=xmlGetProp(nodePtr,(const xmlChar *)"x");
		if(!xmlString)
			return false;
		tmpStr=(char *)xmlString;
		xmlFree(xmlString);

		//Check it is streamable
		if(stream_cast(x,tmpStr))
			return false;

		//--Get Z value--
		xmlString=xmlGetProp(nodePtr,(const xmlChar *)"y");
		if(!xmlString)
			return false;
		tmpStr=(char *)xmlString;
		xmlFree(xmlString);

		//Check it is streamable
		if(stream_cast(y,tmpStr))
			return false;

		//--Get Y value--
		xmlString=xmlGetProp(nodePtr,(const xmlChar *)"z");
		if(!xmlString)
			return false;
		tmpStr=(char *)xmlString;
		xmlFree(xmlString);

		//Check it is streamable
		if(stream_cast(z,tmpStr))
			return false;

		vectorParams.push_back(Point3D(x,y,z));
	}

	return true;
}

bool parseXMLColour(xmlNodePtr &nodePtr, ColourRGBAf &rgba)
{
	xmlChar *xmlString;

	float r,g,b,a;
	std::string tmpStr;
	//--red--
	xmlString=xmlGetProp(nodePtr,(const xmlChar *)"r");
	if(!xmlString)
		return false;
	tmpStr=(char *)xmlString;

	//convert from string to digit
	if(stream_cast(r,tmpStr))
		return false;

	//disallow negative or values gt 1.
	if(r < 0.0f || r > 1.0f)
		return false;
	xmlFree(xmlString);


	//--green--
	xmlString=xmlGetProp(nodePtr,(const xmlChar *)"g");
	if(!xmlString)
		return false;
	tmpStr=(char *)xmlString;

	//convert from string to digit
	if(stream_cast(g,tmpStr))
	{
		xmlFree(xmlString);
		return false;
	}
	
	xmlFree(xmlString);

	//disallow negative or values gt 1.
	if(g < 0.0f || g > 1.0f)
		return false;

	//--blue--
	xmlString=xmlGetProp(nodePtr,(const xmlChar *)"b");
	if(!xmlString)
		return false;
	tmpStr=(char *)xmlString;

	//convert from string to digit
	if(stream_cast(b,tmpStr))
	{
		xmlFree(xmlString);
		return false;
	}
	xmlFree(xmlString);
	
	//disallow negative or values gt 1.
	if(b < 0.0f || b > 1.0f)
		return false;

	//--Alpha--
	xmlString=xmlGetProp(nodePtr,(const xmlChar *)"a");
	if(!xmlString)
		return false;
	tmpStr=(char *)xmlString;

	//convert from string to digit
	if(stream_cast(a,tmpStr))
	{
		xmlFree(xmlString);
		return false;
	}
	xmlFree(xmlString);

	//disallow negative or values gt 1.
	if(a < 0.0f || a > 1.0f)
		return false;
	rgba.r(r);
	rgba.g(g);
	rgba.b(b);
	rgba.a(a);

	return true;
}

const RangeFile *getRangeFile(const std::vector<const FilterStreamData*> &dataIn)
{
	for(size_t ui=0;ui<dataIn.size();ui++)
	{
		if(dataIn[ui]->getStreamType() == STREAM_TYPE_RANGE)
			return ((const RangeStreamData*)(dataIn[ui]))->rangeFile;
	}

	ASSERT(false);
}

size_t getTotalSizeByType(const std::vector<const FilterStreamData*> &dataIn, unsigned int streamTypeMask)
{
	size_t totalSize=0;
	for (size_t i = 0; i < dataIn.size(); i++)
	{
		if(dataIn[i]->getStreamType() & streamTypeMask)
		    totalSize+=dataIn[i]->getNumBasicObjects();
	}
	return totalSize;
}



DrawColourBarOverlay *makeColourBar(float minV, float maxV,size_t nColours,size_t colourMap, bool reverseMap, float alpha) 
{
	//Set up the colour bar. Place it in a draw stream type
	DrawColourBarOverlay *dc = new DrawColourBarOverlay;

	vector<float> r,g,b;
	r.resize(nColours);
	g.resize(nColours);
	b.resize(nColours);

	for (unsigned int ui=0;ui<nColours;ui++)
	{
		unsigned char rgb[3]; //RGB array
		float value;
		value = (float)(ui)*(maxV-minV)/(float)nColours + minV;
		//Pick the desired colour map
		colourMapWrap(colourMap,rgb,value,minV,maxV,reverseMap);
		r[ui]=rgb[0]/255.0f;
		g[ui]=rgb[1]/255.0f;
		b[ui]=rgb[2]/255.0f;
	}

	dc->setColourVec(r,g,b);

	dc->setSize(0.08,0.6);
	dc->setPosition(0.1,0.1);
	dc->setMinMax(minV,maxV);
	dc->setAlpha(alpha);

	return dc;
}

//creates a temporary filename for use
std::string createTmpFilename(const char *dir,const char *extension)
{
	wxString tmpFilename,tmpDir;
	
	if(!dir)
	{
		tmpDir=wxFileName::GetTempDir();


	#if defined(__WIN32__) || defined(__WIN64__)
		tmpDir=tmpDir + wxT("\\3Depict\\");

	#else
		tmpDir=tmpDir + wxT("/3Depict/");
	#endif

	}
	else
		tmpDir=dir;
	
	if(!wxDirExists(tmpDir))
		wxMkdir(tmpDir);
	tmpFilename=wxFileName::CreateTempFileName(tmpDir+ wxT("unittest-"));
	wxRemoveFile(tmpFilename);
	if(extension)
		tmpFilename+=wxT(".pos");

	return stlStr(tmpFilename);
}


std::string transferFunctionString(const std::vector<pair<float,ColourRGBA> >  &colWithOffset)
{
	vector<pair<float,ColourRGBA> > tmp;
	tmp=colWithOffset;
	//Ensure the transfer function x-value is monotonically increasing
	ComparePairFirst cmpF;
	std::sort(tmp.begin(),tmp.end(),cmpF);

	std::string res,tmpS;
	for(unsigned int ui=0; ui<tmp.size();ui++)
	{
		stream_cast(tmpS,tmp[ui].first);
		res+=tmpS+ ",";
		for(unsigned int uj=0;uj<4;uj++)
		{
			stream_cast(tmpS,tmp[ui].second[uj]);
			res+=tmpS;

			if(uj !=3)
				res+=",";
		}
	
		if(ui != colWithOffset.size()-1)
			res+="|";
	}

	return res;
}

bool fromTransferFunctionString(const std::string &s , std::vector<pair<float,ColourRGBA> > &colWithOffset)
{
	vector<string> components;
	splitStrsRef(s.c_str(),'|',components);

	if(!components.size())
		return false;

	pair<float,ColourRGBA> colPair;
	for(unsigned int ui=0;ui<components.size();ui++)
	{
		vector<string> thisCompStrVec;
		splitStrsRef(components[ui].c_str(),',',thisCompStrVec);

		//Should be 5 vector
		if(thisCompStrVec.size() %5)
			return false;

		for(unsigned int uj=0;uj<thisCompStrVec.size(); uj++)
		{
			unsigned int u;
			if(uj == 0)
			{
				if(stream_cast(colPair.first,thisCompStrVec[uj]))
					return false;
			}
			else
			{
				if(stream_cast(u,thisCompStrVec[uj]))
					return false;
				colPair.second[uj-1] = u;
			}

		}
		colWithOffset.push_back(colPair);

	}

	return true;
}

#ifdef DEBUG
bool testCommon()
{
	//IMPLEMENT ME
	return true;
}
#endif
