/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "cdw_task.h"
#include "cdw_regex_dispatch.h"
#include "cdw_processwin.h"
#include "cdw_mkisofs_regex.h"
#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_logging.h"
#include "gettext.h"

extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];

/* time captured at the beginning of process */
extern time_t time0;

extern cdw_task_t *thread_task;


/* functions used for stdout output handling */
/* none */


/* functions used for stderr output handling */
static void handle_bad_option(regex_t *regex, regmatch_t *matches);
static int cdw_mkisofs_handle_image_creation(regex_t *regex, regmatch_t *matches);
static int cdw_mkisofs_handle_image_creation_100percent(regex_t *regex, regmatch_t *matches);
static int cdw_mkisofs_handle_implementation_botch(regex_t *regex, regmatch_t *matches);

static regex_t eregex3, eregex4, eregex5, eregex6;
static regex_t eregex7, eregex8, eregex9, eregex10, eregex11, eregex12;
static regex_t eregex13, eregex14, eregex15, eregex16;
static regmatch_t *ematches3 = NULL, *ematches4 = NULL;
static regmatch_t *ematches5 = NULL, *ematches6 = NULL, *ematches7 = NULL, *ematches8 = NULL;
static regmatch_t *ematches9 = NULL, *ematches10 = NULL, *ematches11 = NULL, *ematches12 = NULL;
static regmatch_t *ematches13 = NULL, *ematches14 = NULL, *ematches15 = NULL, *ematches16 = NULL;


static cdw_regex_t stderr_regex[] = {
	{ "creating 0-99", 301,
	  /* Mkisofs process during image creation (0 - 99,9%) */
	  "([ ]*)([0-9]+).([0-9]+)% done,",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkisofs_handle_image_creation },

	{ "creating 100", 302,
	  /* Mkisofs 100% (image created) */
	  "Total translation table size",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkisofs_handle_image_creation_100percent },

	{ "Implementation botch", 317,
	  /* "/home/acerion/bin/mkisofs: Implementation botch: Hashed file '/my_link_VLT' has illegal inode 5B520.
	      :-( /home/acerion/bin/mkisofs has failed: 255 "

	      This happened when I tried to burn files with growisofs+mkisofs
	      to open DVD disc to which I have already burned some files with xorriso */
	  "Implementation botch: (.+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_mkisofs_handle_implementation_botch },

	{ "table guard", -1,
	  /* guard: debug_id = -1 */
	  "",
	  (regex_t *) NULL, (regmatch_t *) NULL, 0,
	  (cdw_regex_handler_t) NULL }
};





void mkisofs_stderr_regexp_prepare(void)
{
	cdw_rv_t crv = cdw_regex_prepare_regexes_table(stderr_regex);
	if (crv != CDW_OK) {
		cdw_vdm("ERROR: failed to prepare regexes table for stderr\n");
	}

	int rv = 0;

	/* "/usr/bin/mkisofs: Error: <filename1> and <filename2> have the same Joliet name" */
	rv = regcomp(&eregex3, "(have the same Joliet name)", REG_EXTENDED);
	ematches3 = (regmatch_t *) calloc(1, (eregex3.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex3, 33);
	}

	/* "/usr/bin/mkisofs: Joliet tree sort failed." */
	/* "have the same Joliet name" and "Joliet tree sort failed"
	   are somehow related, but I'm not sure if there is 100% correlation,
	   therefore I'm adding this as separate error; perhaps tree sort
	   may fail for other reasons as well */
	rv = regcomp(&eregex4, "tree sort failed", REG_EXTENDED);
	ematches4 = (regmatch_t *) calloc(1, (eregex4.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex4, 34);
	}

	/* "/usr/bin/mkisofs: Uh oh, I cant find the boot image '/home/acerion/boot.img' !" */
	/* File exists, but it is in incorrect location - path to the file
	   should be relative to path specified as main argument to mkisofs */
	rv = regcomp(&eregex5, "can([']*)t find the boot image", REG_EXTENDED);
	ematches5 = (regmatch_t *) calloc(1, (eregex5.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex5, 35);
	}

	/* "/usr/bin/mkisofs: No such file or directory. Invalid node - 'just'." */
	/* "genisoimage: No such file or directory. Invalid node - 'mieta' " */
	/* most probably invalid entry in "other mkisofs options" */
	rv = regcomp(&eregex6, "(mkisofs|genisoimage): No such file or directory", REG_EXTENDED);
	ematches6 = (regmatch_t *) calloc(1, (eregex6.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex6, 36);
	}

	/* "Bad Option '-joiet-long' (error -1 BADFLAG)." */
	/* most probably invalid entry in "other mkisofs options" */
	rv = regcomp(&eregex7, "Bad Option ('.+')", REG_EXTENDED);
	ematches7 = (regmatch_t *) calloc(1, (eregex7.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex7, 37);
	}

	/* "genisoimage: Inode number overflow, too many files in file system" */
	rv = regcomp(&eregex8, "Inode number overflow, too many files in file system", REG_EXTENDED);
	ematches8 = (regmatch_t *) calloc(1, (eregex8.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex8, 38);
	}

	/* "mkisofs: No space left on device" */
	rv = regcomp(&eregex9, "(mkisofs|genisoimage): No space left on device", REG_EXTENDED);
	ematches9 = (regmatch_t *) calloc(1, (eregex9.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex9, 39);
	}

	/* /home/acerion/bin/mkisofs: Old session has illegal name 'AAAAAAAAAAAAAAAAAAAAAAAAAAA.FLV;1' length 33
	   /home/acerion/bin/mkisofs: New session will use    name 'AAAAAAAA.FLV;1'

	   most probably an effect of mismatch in iso level between filesystem
	   already existing on CD and new filesystem created by mkisofs to
	   append to existing one */
	rv = regcomp(&eregex10, "ld session has illegal name", REG_EXTENDED);
	ematches10 = (regmatch_t *) calloc(1, (eregex10.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex10, 310);
	}

	/* Error: '/rr_moved/util' and '/rr_moved/util' have the same Rock Ridge name */
	rv = regcomp(&eregex11, "Error: .+ and .+ have the same Rock Ridge name", REG_EXTENDED);
	ematches11 = (regmatch_t *) calloc(1, (eregex11.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex11, 311);
	}

	/* /home/acerion/bin/mkisofs: File '/home/acerion/current/cdw/working_copy/cdw/src/a' did grow.
	   /home/acerion/bin/mkisofs: Error is considered fatal, aborting. */
	rv = regcomp(&eregex12, "did grow", REG_EXTENDED);
	ematches12 = (regmatch_t *) calloc(1, (eregex12.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex12, 312);
	}

	/* /home/acerion/bin/mkisofs: File '/home/acerion/current/cdw/working_copy/cdw/src/a' did grow.
	   /home/acerion/bin/mkisofs: Error is considered fatal, aborting. */
	rv = regcomp(&eregex13, "Error is considered fatal", REG_EXTENDED);
	ematches13 = (regmatch_t *) calloc(1, (eregex13.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex13, 313);
	}

	/* File /home/acerion/image_4GB.iso is larger than 4GiB-1.
	   -allow-limited-size was not specified. There is no way do represent this file size. Aborting. */
	/* happens on 32-bit machine when selecting single file larger
	   than 4GiB-1 */
	rv = regcomp(&eregex14, "-allow-limited-size was not specified", REG_EXTENDED);
	ematches14 = (regmatch_t *) calloc(1, (eregex14.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex14, 314);
	}

	/* "Value too large for defined data type. File /tmp/image.iso is too
	   large for current mode - ignoring" - occurred when trying to burn file larger
	   than 4GiB with iso level = 1; files that large are allowed by iso level 3;

	   this is what mkisofs reports; genisoimage will rather report this:

	   "File /tmp/image.iso is larger than 4GiB-1.
	   -allow-limited-size was not specified. There is no way do represent this file size. Aborting." */
	rv = regcomp(&eregex15, "is too large for current mode", REG_EXTENDED);
	ematches15 = (regmatch_t *) calloc(1, (eregex15.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex15, 315);
	}

	/* "/home/acerion/bin/mkisofs: Short read on old image
	    :-( /home/acerion/bin/mkisofs has failed: 255"

	   happened when I burned first session to brand new empty DVD-RW,
	   did not reload disc, and attempted to burn second session */
	rv = regcomp(&eregex16, "Short read on old image", REG_EXTENDED);
	ematches16 = (regmatch_t *) calloc(1, (eregex16.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex16, 316);
	}

	return;
}





void mkisofs_stderr_regexp_execute(void)
{
	stderr_pipe_buffer[PIPE_BUFFER_SIZE] = '\0';
	cdw_regex_execute_regexes_table(stderr_regex, stderr_pipe_buffer);

	int rv = 0;

	rv = regexec(&eregex3, stderr_pipe_buffer, eregex3.re_nsub + 1, ematches3, 0);
	if (rv == 0) {
		/* this can happen when using mkisofs for creating iso image,
		   both piped to cdrecord  or growisofs and written to hdd */
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME\n");
	}

	rv = regexec(&eregex4, stderr_pipe_buffer, eregex4.re_nsub + 1, ematches4, 0);
	if (rv == 0) {
		/* this can happen when using mkisofs for creating iso image,
		   both piped to cdrecord or growisofs and written to hdd */
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_JOLIET_TREE_SORT;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_JOLIET_TREE_SORT\n");
	}

	rv = regexec(&eregex5, stderr_pipe_buffer, eregex5.re_nsub + 1, ematches5, 0);
	if (rv == 0) {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_CANT_FIND_BOOT_IMAGE;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_CANT_FIND_BOOT_IMAGE\n");
	}

	rv = regexec(&eregex6, stderr_pipe_buffer, eregex6.re_nsub + 1, ematches6, 0);
	if (rv == 0) {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR\n");
	}

	rv = regexec(&eregex7, stderr_pipe_buffer, eregex7.re_nsub + 1, ematches7, 0);
	if (rv == 0) {
		cdw_sdm ("INFO: calling handle_bad_option()\n");
		handle_bad_option(&eregex7, ematches7);
	}

	rv = regexec(&eregex8, stderr_pipe_buffer, eregex8.re_nsub + 1, ematches8, 0);
	if (rv == 0) {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_INODE_NUMBER_OVERFLOW;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_INODE_NUMBER_OVERFLOW\n");
	}

	rv = regexec(&eregex9, stderr_pipe_buffer, eregex9.re_nsub + 1, ematches9, 0);
	if (rv == 0) {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_NO_SPACE_LEFT;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_NO_SPACE_LEFT\n");
	}

	rv = regexec(&eregex10, stderr_pipe_buffer, eregex10.re_nsub + 1, ematches10, 0);
	if (rv == 0) {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_ISO_LEVEL_MISMATCH;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_ISO_LEVEL_MISMATCH\n");
	}

	rv = regexec(&eregex11, stderr_pipe_buffer, eregex11.re_nsub + 1, ematches11, 0);
	if (rv == 0) {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME\n");
	}

	rv = regexec(&eregex12, stderr_pipe_buffer, eregex12.re_nsub + 1, ematches12, 0);
	if (rv == 0) {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW\n");
	}

	rv = regexec(&eregex13, stderr_pipe_buffer, eregex13.re_nsub + 1, ematches13, 0);
	if (rv == 0) {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_FATAL_ERROR;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_FATAL_ERROR\n");
	}

	/* "-allow-limited-size was not specified" */
	rv = regexec(&eregex14, stderr_pipe_buffer, eregex14.re_nsub + 1, ematches14, 0);
	if (rv == 0) {
	        thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_ALLOW_LIMITED_SIZE;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_ALLOW_LIMITED_SIZE\n");
	}

	/* "is too large for current mode" */
	rv = regexec(&eregex15, stderr_pipe_buffer, eregex15.re_nsub + 1, ematches15, 0);
	if (rv == 0) {
	        thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_LOW_ISO_LEVEL;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_LOW_ISO_LEVEL\n");
	}

	/* "Short read on old image" */
	rv = regexec(&eregex16, stderr_pipe_buffer, eregex16.re_nsub + 1, ematches16, 0);
	if (rv == 0) {
	        thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_SHORT_READ;
		cdw_vdm ("ERROR: tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_SHORT_READ\n");
	}

	return;
}





void mkisofs_stderr_regexp_destroy(void)
{
	cdw_regex_clean_up_regexes_table(stderr_regex);

	regfree(&eregex3);
	regfree(&eregex4);
	regfree(&eregex5);
	regfree(&eregex6);
	regfree(&eregex7);
	regfree(&eregex8);
	regfree(&eregex9);
	regfree(&eregex10);
	regfree(&eregex11);
	regfree(&eregex12);
	regfree(&eregex13);
	regfree(&eregex14);
	regfree(&eregex15);
	regfree(&eregex16);

	free(ematches3);
	free(ematches4);
	free(ematches5);
	free(ematches6);
	free(ematches7);
	free(ematches8);
	free(ematches9);
	free(ematches10);
	free(ematches11);
	free(ematches12);
	free(ematches13);
	free(ematches14);
	free(ematches15);
	free(ematches16);

	ematches3 = (regmatch_t *) NULL;
	ematches4 = (regmatch_t *) NULL;
	ematches5 = (regmatch_t *) NULL;
	ematches6 = (regmatch_t *) NULL;
	ematches7 = (regmatch_t *) NULL;
	ematches8 = (regmatch_t *) NULL;
	ematches9 = (regmatch_t *) NULL;
	ematches10 = (regmatch_t *) NULL;
	ematches11 = (regmatch_t *) NULL;
	ematches12 = (regmatch_t *) NULL;
	ematches13 = (regmatch_t *) NULL;
	ematches14 = (regmatch_t *) NULL;
	ematches15 = (regmatch_t *) NULL;
	ematches16 = (regmatch_t *) NULL;

	return;
}



/* these three functions are currently unused, so it is ok that they are empty */

void mkisofs_stdout_regexp_prepare(void)
{
	return;
}


void mkisofs_stdout_regexp_execute(void)
{
	return;
}


void mkisofs_stdout_regexp_destroy(void)
{
	return;
}





/**
   \brief Handle normal mkisofs output during image creation

   Normal mkisofs output during image creation looks like this:
   "58.65% done, estimate finish Sat Mar  3 21:20:06 2007"
   This function catches value of "done" (discarding the part
   after decimal point) and represents it in processwin.

   Note mkisofs doesn't inform explicitly when "done" has reached
   100%, therefore handle_image_creation_100percent() is needed.

   \param regex
   \param matches
*/
int cdw_mkisofs_handle_image_creation(regex_t *regex, regmatch_t *matches)
{
	/* additional check is to avoid mistake with direct writing */
	if (thread_task->id != CDW_TASK_CREATE_IMAGE_ISO9660) {
		return 0;
	}

	/* Regexp for capturing information on creating image looks like this:
	     1      2        3
	  "([ ]*)([0-9]+).([0-9]+)% done,"
	  I am interested only in subexpression 2 - decimal part of value;
	  I could also capture and process fractional part, but I don't
	  see much need to do so */
	cdw_regex_assert_subex_number(regex->re_nsub, 3);

	const unsigned int sub = 2;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	/* amount of data written to image */
	int done_size = atoi((char *) &submatch);

	double time_from_start = difftime(time(NULL), time0);
	int total_size = 100;
	int todo_size = total_size - done_size;
	int eta = (int) (((1.0 * todo_size) / done_size) * time_from_start);

	char eta_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	memset(eta_string, '\0', PROCESSWIN_MAX_RTEXT_LEN + 1);
	cdw_utils_eta_calculations(eta_string, eta, PROCESSWIN_MAX_RTEXT_LEN);

	cdw_processwin_display_eta(eta_string);
	cdw_processwin_display_progress_conditional(done_size, 100, (char *) NULL);
	cdw_processwin_wrefresh();

	return 0;
}





/**
   \brief Handle normal mkisofs output after image creation

   Normal mkisofs output during image creation like this:

   '58.65% done, estimate finish Sat Mar  3 21:20:06 2007'.

   mkisofs doesn't write 100% value, but it prints following after completion:

   '
   Total translation table size: 0
   Total rockridge attributes bytes: 0
   Total directory bytes: 0
   Path table size(bytes): 10
   Max brk space used 0
   127880 extents written (249 MB)
   '

   This function is called after capturing "Total translation table size"
*/
int cdw_mkisofs_handle_image_creation_100percent(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* additional check is to avoid mistake with direct writing */
	if (thread_task->id != CDW_TASK_CREATE_IMAGE_ISO9660) {
		return 0;
	}

	/* fake some data to show that creating image has finished */
	char eta_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	memset(eta_string, '\0', PROCESSWIN_MAX_RTEXT_LEN + 1);
	cdw_utils_eta_calculations(eta_string, 0, PROCESSWIN_MAX_RTEXT_LEN); /* ETA == 0 */

	cdw_processwin_display_eta(eta_string);
	cdw_processwin_display_progress(100, 100, (char *) NULL);
	cdw_processwin_wrefresh();

	return 0;
}





/**
   \brief Handle situation when mkisofs complains about bad command line option

   When mkisofs is provided with incorrect command line option, it prints
   message like this: "Bad Option '-joiet-long' (error -1 BADFLAG)."

   Options supplied to mkisofs are set either by checkboxes and input fields
   that are validated, or in "other mkisofs options" input field that
   is not validated for correctness of mkisofs options. Error reported by
   mkisofs and captured by regexp is probably a result of invalid string in
   "other mkisofs options" field.

   \param regex
   \param matches
*/
void handle_bad_option(regex_t *regex, regmatch_t *matches)
{
	/* regexp for bad option is
                         1
	   "Bad Option ('.+')"
	   so we need to catch subexpression 1 */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	const unsigned int sub = 1;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
	} else {
		cdw_vdm ("ERROR: invalid mkisofs option is \"%s\"\n", submatch);
	}
	thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_BAD_OPTION;
	cdw_vdm ("ERROR: thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_BAD_OPTION\n");

	return;
}





int cdw_mkisofs_handle_implementation_botch(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing error:
                                    1
	   "Implementation botch: (.+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	unsigned int sub = 1;
	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, sub, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);

		return -1;
	} else {
		thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_IMPLEMENTATION_BOTCH;
		cdw_vdm ("ERROR: thread_task->tool_status.mkisofs |= CDW_TOOL_STATUS_MKISOFS_IMPLEMENTATION_BOTCH\n");
		/* 2TRANS: this is message printed to log file */
		cdw_logging_write(_("ERROR: problem with creating file system on optical disc.\n"));
		/* 2TRANS: this is message printed to log file */
		cdw_logging_write(_("ERROR: Perhaps this information will help resolve the problem:\n"));
		/* 2TRANS: this is message printed to log file;
		   '%s' is an error message printed by mkisofs */
		cdw_logging_write(_("ERROR: %s\n"), submatch);

		return 0;
	}
}
