// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/system/privacy_hub/geolocation_privacy_switch_controller.h"

#include <string>

#include "ash/constants/ash_pref_names.h"
#include "ash/constants/geolocation_access_level.h"
#include "ash/public/cpp/session/session_observer.h"
#include "ash/session/session_controller_impl.h"
#include "ash/shell.h"
#include "ash/system/privacy_hub/privacy_hub_controller.h"
#include "ash/system/privacy_hub/privacy_hub_metrics.h"
#include "ash/system/privacy_hub/privacy_hub_notification_controller.h"
#include "base/notreached.h"
#include "components/prefs/pref_change_registrar.h"
#include "components/prefs/pref_service.h"

namespace ash {

GeolocationPrivacySwitchController::GeolocationPrivacySwitchController()
    : session_observation_(this) {
  session_observation_.Observe(Shell::Get()->session_controller());
}

GeolocationPrivacySwitchController::~GeolocationPrivacySwitchController() =
    default;

// static
GeolocationPrivacySwitchController* GeolocationPrivacySwitchController::Get() {
  PrivacyHubController* privacy_hub_controller =
      Shell::Get()->privacy_hub_controller();
  return privacy_hub_controller
             ? privacy_hub_controller->geolocation_controller()
             : nullptr;
}

void GeolocationPrivacySwitchController::OnActiveUserPrefServiceChanged(
    PrefService* pref_service) {
  pref_change_registrar_ = std::make_unique<PrefChangeRegistrar>();
  pref_change_registrar_->Init(pref_service);
  pref_change_registrar_->Add(
      prefs::kUserGeolocationAccessLevel,
      base::BindRepeating(
          &GeolocationPrivacySwitchController::OnPreferenceChanged,
          base::Unretained(this)));

  if (features::IsCrosPrivacyHubEnabled() &&
      features::IsCrosPrivacyHubLocationEnabled()) {
    UpdateNotification();
  } else {
    // Feature disabled means geolocation is always allowed.
    OnPreferenceChanged();
  }
}

void GeolocationPrivacySwitchController::OnPreferenceChanged() {
  VLOG(1) << "Privacy Hub: Geolocation switch state = "
          << static_cast<int>(AccessLevel());
  if (features::IsCrosPrivacyHubEnabled() &&
      features::IsCrosPrivacyHubLocationEnabled()) {
    UpdateNotification();
  } else {
    // Feature disabled means geolocation is always allowed
    CHECK(pref_change_registrar_);
    pref_change_registrar_->prefs()->SetInteger(
        prefs::kUserGeolocationAccessLevel,
        static_cast<int>(GeolocationAccessLevel::kAllowed));
  }
}

void GeolocationPrivacySwitchController::TrackGeolocationAttempted(
    const std::string& app_name) {
  ++usage_per_app_[app_name];
  ++usage_cnt_;
  UpdateNotification();
}

void GeolocationPrivacySwitchController::TrackGeolocationRelinquished(
    const std::string& app_name) {
  --usage_per_app_[app_name];
  --usage_cnt_;
  if (usage_per_app_[app_name] < 0 || usage_cnt_ < 0) {
    LOG(ERROR) << "Geolocation usage termination without start: count("
               << app_name << ") = " << usage_per_app_[app_name]
               << ", total count = " << usage_cnt_;
    NOTREACHED();
  }

  UpdateNotification();
}

bool GeolocationPrivacySwitchController::IsGeolocationUsageAllowedForApps() {
  switch (AccessLevel()) {
    case GeolocationAccessLevel::kAllowed:
      return true;
    case GeolocationAccessLevel::kOnlyAllowedForSystem:
    case GeolocationAccessLevel::kDisallowed:
      return false;
  }
}

std::vector<std::u16string> GeolocationPrivacySwitchController::GetActiveApps(
    size_t max_count) const {
  std::vector<std::u16string> apps;
  for (const auto& [name, cnt] : usage_per_app_) {
    if (cnt > 0) {
      apps.push_back(base::UTF8ToUTF16(name));
      if (apps.size() == max_count) {
        break;
      }
    }
  }
  return apps;
}

GeolocationAccessLevel GeolocationPrivacySwitchController::AccessLevel() const {
  CHECK(pref_change_registrar_);
  return static_cast<GeolocationAccessLevel>(
      pref_change_registrar_->prefs()->GetInteger(
          prefs::kUserGeolocationAccessLevel));
}

void GeolocationPrivacySwitchController::UpdateNotification() {
  PrivacyHubNotificationController* notification_controller =
      PrivacyHubNotificationController::Get();
  if (!notification_controller) {
    return;
  }

  if (usage_cnt_ == 0 || IsGeolocationUsageAllowedForApps()) {
    notification_controller->RemoveSoftwareSwitchNotification(
        SensorDisabledNotificationDelegate::Sensor::kLocation);
    return;
  }

  notification_controller->ShowSoftwareSwitchNotification(
      SensorDisabledNotificationDelegate::Sensor::kLocation);
}

}  // namespace ash
