// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/webui/shortcut_customization_ui/backend/accelerator_layout_table.h"
#include <string>

#include "ash/public/cpp/accelerators_util.h"
#include "ash/public/mojom/accelerator_info.mojom.h"
#include "ash/strings/grit/ash_strings.h"
#include "base/check_op.h"
#include "base/containers/fixed_flat_map.h"
#include "base/no_destructor.h"
#include "base/notreached.h"
#include "base/strings/utf_string_conversions.h"
#include "chromeos/strings/grit/chromeos_strings.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/events/event_constants.h"
#include "ui/events/keycodes/dom/dom_code.h"
#include "ui/events/keycodes/dom/dom_key.h"
#include "ui/events/keycodes/dom/keycode_converter.h"
#include "ui/events/keycodes/dom_us_layout_data.h"
#include "ui/events/keycodes/keyboard_code_conversion.h"
#include "ui/events/keycodes/keyboard_codes_posix.h"

namespace ash {

// Constructor used for text-based layout accelerators.
NonConfigurableAcceleratorDetails::NonConfigurableAcceleratorDetails(
    int message_id,
    std::vector<TextAcceleratorPart> replacements) {
  this->message_id = message_id;
  this->replacements = std::move(replacements);
}

NonConfigurableAcceleratorDetails::NonConfigurableAcceleratorDetails(
    int resource_id) {
  message_id = resource_id;
}

// Constructor used for standard accelerators (i.e, it contains at least one
// modifier and a set of keys).
NonConfigurableAcceleratorDetails::NonConfigurableAcceleratorDetails(
    std::vector<ui::Accelerator> accels) {
  accelerators = std::move(accels);
}

NonConfigurableAcceleratorDetails::NonConfigurableAcceleratorDetails(
    const NonConfigurableAcceleratorDetails&) = default;
NonConfigurableAcceleratorDetails& NonConfigurableAcceleratorDetails::operator=(
    const NonConfigurableAcceleratorDetails&) = default;

NonConfigurableAcceleratorDetails::~NonConfigurableAcceleratorDetails() =
    default;

const NonConfigurableActionsMap& GetNonConfigurableActionsMap() {
  static base::NoDestructor<NonConfigurableActionsMap>
      nonConfigurableActionsMap({
          // Ambient Accelerator with replacement.
          {NonConfigurableActions::kBrowserSelectTabByIndex,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_GO_TO_TAB_IN_RANGE,
               {TextAcceleratorPart(ui::EF_CONTROL_DOWN),
                TextAcceleratorPart(TextAcceleratorDelimiter::kPlusSign),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_1),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_8)})},
          {NonConfigurableActions::kAmbientOpenLinkInTab,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_OPEN_LINK_IN_TAB,
               {TextAcceleratorPart(ui::EF_CONTROL_DOWN),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientOpenLinkInTabBackground,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_OPEN_LINK_IN_TAB_BACKGROUND,
               {TextAcceleratorPart(ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientOpenLinkInWindow,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_OPEN_LINK_IN_WINDOW,
               {TextAcceleratorPart(ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientOpenPageInNewTab,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_OPEN_PAGE_IN_NEW_TAB,
               {TextAcceleratorPart(ui::EF_ALT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_RETURN)})},
          {NonConfigurableActions::kAmbientCycleForwardMRU,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_CYCLE_FORWARD_MRU,
               {TextAcceleratorPart(ui::EF_ALT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_TAB)})},
          {NonConfigurableActions::kAmbientCycleBackwardMRU,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_CYCLE_BACKWARD_MRU,
               {TextAcceleratorPart(ui::EF_ALT_DOWN),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_TAB)})},
          {NonConfigurableActions::kAmbientLaunchNumberedApp,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_LAUNCH_NUMBERED_APP,
               {TextAcceleratorPart(ui::EF_ALT_DOWN),
                TextAcceleratorPart(TextAcceleratorDelimiter::kPlusSign),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_1),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_8)})},
          {NonConfigurableActions::kAmbientOpenFile,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_OPEN_FILE,
               {TextAcceleratorPart(ui::KeyboardCode::VKEY_SPACE)})},
          {NonConfigurableActions::kAmbientHighlightNextItemOnShelf,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_HIGHLIGHT_NEXT_ITEM_ON_SHELF,
               {TextAcceleratorPart(ui::EF_ALT_DOWN),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_L),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_TAB),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_RIGHT)})},
          {NonConfigurableActions::kAmbientHighlightPreviousItemOnShelf,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_HIGHTLIGHT_PREVIOUS_ITEM_ON_SHELF,
               {TextAcceleratorPart(ui::EF_ALT_DOWN),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_L),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_TAB),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_LEFT)})},
          {NonConfigurableActions::kAmbientOpenHighlightedItemOnShelf,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_OPEN_HIGHLIGHTED_ITEM_ON_SHELF,
               {TextAcceleratorPart(ui::EF_ALT_DOWN),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_L),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_SPACE),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_RETURN)})},
          {NonConfigurableActions::kAmbientRemoveHighlightOnShelf,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_REMOVE_HIGHLIGHT_ON_SHELF,
               {TextAcceleratorPart(ui::EF_ALT_DOWN),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_L),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_ESCAPE)})},
          {NonConfigurableActions::kAmbientMoveAppsInGrid,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_MOVE_APPS_IN_GRID,
               {TextAcceleratorPart(ui::EF_CONTROL_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_LEFT),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_RIGHT),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_UP),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_DOWN)})},
          {NonConfigurableActions::kAmbientMoveAppsInOutFolder,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_MOVE_APPS_IN_OUT_FOLDER,
               {TextAcceleratorPart(ui::EF_CONTROL_DOWN),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_LEFT),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_RIGHT),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_UP),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_DOWN)})},
          {NonConfigurableActions::kAmbientOpenLinkInTabBackground,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_OPEN_LINK_IN_TAB_BACKGROUND,
               {TextAcceleratorPart(ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserStopDragTab,
           NonConfigurableAcceleratorDetails(
               IDS_BROWSER_ACCELERATOR_STOP_DRAG_TAB,
               {TextAcceleratorPart(ui::KeyboardCode::VKEY_ESCAPE)})},
          {NonConfigurableActions::kAmbientActivateIndexedDesk,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_ACTIVATE_INDEXED_DESK,
               {TextAcceleratorPart(ui::EF_COMMAND_DOWN),
                TextAcceleratorPart(ui::EF_SHIFT_DOWN),
                TextAcceleratorPart(TextAcceleratorDelimiter::kPlusSign),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_1),
                TextAcceleratorPart(ui::KeyboardCode::VKEY_8)})},
          // Ambient accelerators that only contain plain text (no
          // replacements).
          {NonConfigurableActions::kAmbientDragLinkInNewTab,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_DRAG_LINK_IN_NEW_TAB)},
          {NonConfigurableActions::kAmbientDragLinkInSameTab,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_DRAG_LINK_IN_SAME_TAB)},
          {NonConfigurableActions::kAmbientSaveLinkAsBookmark,
           NonConfigurableAcceleratorDetails(
               IDS_AMBIENT_ACCELERATOR_SAVE_LINK_AS_BOOKMARK)},
          // Standard accelerators.
          {NonConfigurableActions::kBrowserNewTab,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_T, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserCloseTab,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_W, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserCloseWindow,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_W, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserSelectLastTab,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_9, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserOpenFile,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_O, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserNewIncognitoWindow,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_N, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserNewWindow,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_N, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserRestoreTab,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_T, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserTabSearch,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_A, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserClearBrowsingData,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_BACK, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserCloseFindOrStop,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_ESCAPE, ui::EF_NONE)})},
          {NonConfigurableActions::kBrowserFocusBookmarks,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_B, ui::EF_SHIFT_DOWN | ui::EF_ALT_DOWN)})},
          {NonConfigurableActions::kBrowserBack,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_LEFT, ui::EF_ALT_DOWN)})},
          {NonConfigurableActions::kBrowserForward,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_RIGHT, ui::EF_ALT_DOWN)})},
          {NonConfigurableActions::kBrowserFind,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_F, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserShowDownloads,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_J, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserShowHistory,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_H, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserFocusMenuBar,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_F10, ui::EF_NONE)})},
          {NonConfigurableActions::kBrowserPrint,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_P, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserReloadBypassingCache,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_R, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserZoomNormal,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_0, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserBookmarkAllTabs,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_D, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserSavePage,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_S, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserBookmarkThisTab,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_D, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserShowBookmarkManager,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_O, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserDevToolsConsole,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_J, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserDevToolsInspect,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_C, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserDevTools,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_I, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserShowBookmarkBar,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_B, ui::EF_SHIFT_DOWN | ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserViewSource,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_U, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserZoomMinus,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_OEM_MINUS, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserZoomPlus,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_OEM_PLUS, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserFocusToolbar,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_T, ui::EF_SHIFT_DOWN | ui::EF_ALT_DOWN)})},
          {NonConfigurableActions::kBrowserFocusInactivePopupForAccessibility,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_A, ui::EF_SHIFT_DOWN | ui::EF_ALT_DOWN)})},
          {NonConfigurableActions::kBrowserBottomPage,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_END, ui::EF_NONE)})},
          {NonConfigurableActions::kBrowserTopPage,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_HOME, ui::EF_NONE)})},
          {NonConfigurableActions::kBrowserPageUp,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_PRIOR, ui::EF_NONE)})},
          {NonConfigurableActions::kBrowserPageDown,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_NEXT, ui::EF_NONE)})},
          {NonConfigurableActions::kAmbientDeleteNextWord,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_DELETE, ui::EF_NONE)})},
          {NonConfigurableActions::kAmbientGoToBeginningOfLine,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_LEFT, ui::EF_COMMAND_DOWN)})},
          {NonConfigurableActions::kAmbientGoToEndOfLine,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_RIGHT, ui::EF_COMMAND_DOWN)})},
          {NonConfigurableActions::kBrowserNextPane,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_BROWSER_BACK,
                                ui::EF_CONTROL_DOWN | ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientOpenRightClickMenu,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_F10, ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientDisplayHiddenFiles,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_OEM_PERIOD, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientCaretBrowsing,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_F7, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserAutoComplete,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_RETURN, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserHome,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_HOME, ui::EF_ALT_DOWN)})},
          {NonConfigurableActions::kBrowserSelectNextTab,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_TAB, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserSelectPreviousTab,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_TAB, ui::EF_CONTROL_DOWN | ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientCopy,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_C, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientCut,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_X, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientPaste,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_V, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientPastePlainText,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_V, ui::EF_CONTROL_DOWN | ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientDeletePreviousWord,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_BACK, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientUndo,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_Z, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientRedo,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_Z,
                                ui::EF_CONTROL_DOWN | ui::EF_SHIFT_DOWN),
                ui::Accelerator(ui::VKEY_Y, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientContentContextSelectAll,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_A, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientSelectTextToBeginning,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_HOME, ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientSelectTextToEndOfLine,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_END, ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientSelectPreviousWord,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_LEFT, ui::EF_CONTROL_DOWN | ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAMbientSelectNextWord,
           NonConfigurableAcceleratorDetails({ui::Accelerator(
               ui::VKEY_RIGHT, ui::EF_CONTROL_DOWN | ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kAmbientGoToBeginningOfDocument,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_HOME, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientGoToEndOfDocument,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_END, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientMoveStartOfPreviousWord,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_LEFT, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientMoveToEndOfWord,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_RIGHT, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserFindNext,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_G, ui::EF_CONTROL_DOWN),
                // TODO(longbowei): Confirm if we want to keep this accelerator
                // or remove it.
                ui::Accelerator(ui::VKEY_RETURN, ui::EF_NONE)})},
          {NonConfigurableActions::kBrowserFindPrevious,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_G,
                                ui::EF_CONTROL_DOWN | ui::EF_SHIFT_DOWN),
                // TODO(longbowei): Confirm if we want to keep this accelerator
                // or remove it.
                ui::Accelerator(ui::VKEY_RETURN, ui::EF_SHIFT_DOWN)})},
          {NonConfigurableActions::kBrowserFocusAddressBar,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_L, ui::EF_CONTROL_DOWN),
                ui::Accelerator(ui::VKEY_D, ui::EF_ALT_DOWN)})},
          {NonConfigurableActions::kBrowserFocusSearch,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_K, ui::EF_CONTROL_DOWN),
                ui::Accelerator(ui::VKEY_E, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kBrowserShowAppMenu,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_E, ui::EF_ALT_DOWN),
                ui::Accelerator(ui::VKEY_F, ui::EF_ALT_DOWN)})},
          {NonConfigurableActions::kBrowserReload,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_BROWSER_REFRESH, ui::EF_NONE),
                ui::Accelerator(ui::VKEY_R, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientSwitchFocusForwards,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_BROWSER_BACK, ui::EF_CONTROL_DOWN)})},
          {NonConfigurableActions::kAmbientSwitchFocusBackwards,
           NonConfigurableAcceleratorDetails(
               {ui::Accelerator(ui::VKEY_BROWSER_BACK,
                                ui::EF_CONTROL_DOWN | ui::EF_SHIFT_DOWN)})},
      });
  return *nonConfigurableActionsMap;
}

const AcceleratorLayoutMap& GetAcceleratorLayoutMap() {
  static base::NoDestructor<AcceleratorLayoutMap> acceleratorLayoutMap(
      {// General
       {AcceleratorAction::kToggleAppList,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleAppList,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_APP_LIST,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleOverview,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleOverview,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_OVERVIEW,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleSystemTrayBubble,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleSystemTrayBubble,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_SYSTEM_TRAY_BUBBLE,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleCalendar,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleCalendar,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_CALENDAR,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleMessageCenterBubble,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleMessageCenterBubble,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MESSAGE_CENTER_BUBBLE,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kTakeScreenshot,
        AcceleratorLayoutDetails(
            AcceleratorAction::kTakeScreenshot,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TAKE_SCREENSHOT,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kTakePartialScreenshot,
        AcceleratorLayoutDetails(
            AcceleratorAction::kTakePartialScreenshot,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TAKE_PARTIAL_SCREENSHOT,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kTakeWindowScreenshot,
        AcceleratorLayoutDetails(
            AcceleratorAction::kTakeWindowScreenshot,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TAKE_WINDOW_SCREENSHOT,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kStopScreenRecording,
        AcceleratorLayoutDetails(
            AcceleratorAction::kStopScreenRecording,
            IDS_ASH_ACCELERATOR_DESCRIPTION_STOP_SCREEN_RECORDING,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kLockScreen,
        AcceleratorLayoutDetails(
            AcceleratorAction::kLockScreen,
            IDS_ASH_ACCELERATOR_DESCRIPTION_LOCK_SCREEN,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kSuspend,
        AcceleratorLayoutDetails(
            AcceleratorAction::kSuspend,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SUSPEND,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kExit,
        AcceleratorLayoutDetails(
            AcceleratorAction::kExit, IDS_ASH_ACCELERATOR_DESCRIPTION_EXIT,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kSwitchToNextUser,
        AcceleratorLayoutDetails(
            AcceleratorAction::kSwitchToNextUser,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_NEXT_USER,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kSwitchToPreviousUser,
        AcceleratorLayoutDetails(
            AcceleratorAction::kSwitchToPreviousUser,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_PREVIOUS_USER,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kEnableOrToggleDictation,
        AcceleratorLayoutDetails(
            AcceleratorAction::kEnableOrToggleDictation,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_DICTATION,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kStartAssistant,
        AcceleratorLayoutDetails(
            AcceleratorAction::kStartAssistant,
            IDS_ASH_ACCELERATOR_DESCRIPTION_START_ASSISTANT,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kGeneralControls,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kOpenFileManager,
        AcceleratorLayoutDetails(
            AcceleratorAction::kOpenFileManager,
            IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_FILE_MANAGER,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kAmbientOpenFile,
        AcceleratorLayoutDetails(NonConfigurableActions::kAmbientOpenFile,
                                 IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_FILE,
                                 mojom::AcceleratorCategory::kGeneral,
                                 mojom::AcceleratorSubcategory::kApps,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kText,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientDisplayHiddenFiles,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientDisplayHiddenFiles,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DISPLAY_HIDDEN_FILES,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kShowShortcutViewer,
        AcceleratorLayoutDetails(
            AcceleratorAction::kShowShortcutViewer,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_SHORTCUT_VIEWER,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kOpenCalculator,
        AcceleratorLayoutDetails(
            AcceleratorAction::kOpenCalculator,
            IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_CALCULATOR,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kOpenDiagnostics,
        AcceleratorLayoutDetails(
            AcceleratorAction::kOpenDiagnostics,
            IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_DIAGNOSTICS,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kOpenGetHelp,
        AcceleratorLayoutDetails(AcceleratorAction::kOpenGetHelp,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_GET_HELP,
                                 mojom::AcceleratorCategory::kGeneral,
                                 mojom::AcceleratorSubcategory::kApps,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kOpenFeedbackPage,
        AcceleratorLayoutDetails(
            AcceleratorAction::kOpenFeedbackPage,
            IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_FEEDBACK_PAGE,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kAmbientLaunchNumberedApp,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientLaunchNumberedApp,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kLaunchLastApp,
        AcceleratorLayoutDetails(
            AcceleratorAction::kLaunchLastApp,
            IDS_ASH_ACCELERATOR_DESCRIPTION_LAUNCH_LAST_APP,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleResizeLockMenu,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleResizeLockMenu,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_RESIZE_LOCK_MENU,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kShowTaskManager,
        AcceleratorLayoutDetails(
            AcceleratorAction::kShowTaskManager,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_TASK_MANAGER,
            mojom::AcceleratorCategory::kGeneral,
            mojom::AcceleratorSubcategory::kApps,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kOpenCrosh,
        AcceleratorLayoutDetails(AcceleratorAction::kOpenCrosh,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_CROSH,
                                 mojom::AcceleratorCategory::kGeneral,
                                 mojom::AcceleratorSubcategory::kApps,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},

       // Device
       {AcceleratorAction::kVolumeUp,
        AcceleratorLayoutDetails(AcceleratorAction::kVolumeUp,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_UP,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kMedia,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kVolumeDown,
        AcceleratorLayoutDetails(AcceleratorAction::kVolumeDown,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_DOWN,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kMedia,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kVolumeMute,
        AcceleratorLayoutDetails(AcceleratorAction::kVolumeMute,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_MUTE,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kMedia,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMicrophoneMuteToggle,
        AcceleratorLayoutDetails(
            AcceleratorAction::kMicrophoneMuteToggle,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MICROPHONE_MUTE,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kMedia,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMediaPlay,
        AcceleratorLayoutDetails(AcceleratorAction::kMediaPlay,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_PLAY_MEDIA,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kMedia,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMediaPause,
        AcceleratorLayoutDetails(AcceleratorAction::kMediaPause,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_PAUSE_MEDIA,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kMedia,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMediaPlayPause,
        AcceleratorLayoutDetails(
            AcceleratorAction::kMediaPlayPause,
            IDS_ASH_ACCELERATOR_DESCRIPTION_PLAY_PAUSE_MEDIA,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kMedia,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMediaNextTrack,
        AcceleratorLayoutDetails(AcceleratorAction::kMediaNextTrack,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_NEXT_TRACK,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kMedia,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMediaPrevTrack,
        AcceleratorLayoutDetails(AcceleratorAction::kMediaPrevTrack,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_PREVIOUS_TRACK,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kMedia,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMediaFastForward,
        AcceleratorLayoutDetails(
            AcceleratorAction::kMediaFastForward,
            IDS_ASH_ACCELERATOR_DESCRIPTION_FAST_FORWARD_MEDIA,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kMedia,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kFocusPip,
        AcceleratorLayoutDetails(AcceleratorAction::kFocusPip,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_PIP,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kMedia,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kKeyboardBacklightToggle,
        AcceleratorLayoutDetails(
            AcceleratorAction::kKeyboardBacklightToggle,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_KEYBOARD_BACKLIGHT,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kInputs,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kKeyboardBrightnessUp,
        AcceleratorLayoutDetails(
            AcceleratorAction::kKeyboardBrightnessUp,
            IDS_ASH_ACCELERATOR_DESCRIPTION_KEYBOARD_BRIGHTNESS_UP,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kInputs,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kKeyboardBrightnessDown,
        AcceleratorLayoutDetails(
            AcceleratorAction::kKeyboardBrightnessDown,
            IDS_ASH_ACCELERATOR_DESCRIPTION_KEYBOARD_BRIGHTNESS_DOWN,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kInputs,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleImeMenuBubble,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleImeMenuBubble,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_IME_MENU_BUBBLE,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kInputs,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kSwitchToNextIme,
        AcceleratorLayoutDetails(
            AcceleratorAction::kSwitchToNextIme,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_NEXT_IME,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kInputs,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kSwitchToLastUsedIme,
        AcceleratorLayoutDetails(
            AcceleratorAction::kSwitchToLastUsedIme,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_LAST_USED_IME,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kInputs,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleStylusTools,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleStylusTools,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_STYLUS_TOOLS,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kInputs,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kBrightnessUp,
        AcceleratorLayoutDetails(AcceleratorAction::kBrightnessUp,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_BRIGHTNESS_UP,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kDisplay,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kBrightnessDown,
        AcceleratorLayoutDetails(
            AcceleratorAction::kBrightnessDown,
            IDS_ASH_ACCELERATOR_DESCRIPTION_BRIGHTNESS_DOWN,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kDisplay,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kScaleUiUp,
        AcceleratorLayoutDetails(AcceleratorAction::kScaleUiUp,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_UP,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kDisplay,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kScaleUiDown,
        AcceleratorLayoutDetails(AcceleratorAction::kScaleUiDown,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_DOWN,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kDisplay,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kScaleUiReset,
        AcceleratorLayoutDetails(AcceleratorAction::kScaleUiReset,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_RESET,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kDisplay,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kPrivacyScreenToggle,
        AcceleratorLayoutDetails(
            AcceleratorAction::kPrivacyScreenToggle,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_PRIVACY_SCREEN,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kDisplay,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleMirrorMode,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleMirrorMode,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MIRROR_MODE,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kDisplay,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kSwapPrimaryDisplay,
        AcceleratorLayoutDetails(
            AcceleratorAction::kSwapPrimaryDisplay,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SWAP_PRIMARY_DISPLAY,
            mojom::AcceleratorCategory::kDevice,
            mojom::AcceleratorSubcategory::kDisplay,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kRotateScreen,
        AcceleratorLayoutDetails(AcceleratorAction::kRotateScreen,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_ROTATE_SCREEN,
                                 mojom::AcceleratorCategory::kDevice,
                                 mojom::AcceleratorSubcategory::kDisplay,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},

       // Browser
       {NonConfigurableActions::kBrowserPrint,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserPrint,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_PRINT,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kGeneral,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserShowAppMenu,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserShowAppMenu,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_APP_MENU,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kGeneral,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserShowDownloads,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserShowDownloads,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_DOWNLOADS,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kGeneral,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserShowHistory,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserShowHistory,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_HISTORY,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kGeneral,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserClearBrowsingData,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserClearBrowsingData,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLEAR_BROWSING_DATA,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kGeneral,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserOpenFile,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserOpenFile,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_OPEN_FILE,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kGeneral,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFocusAddressBar,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserFocusAddressBar,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_ADDRESS_BAR,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBrowserNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFocusSearch,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserFocusSearch,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_SEARCH,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBrowserNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserAutoComplete,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserAutoComplete,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_AUTO_COMPLETE,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBrowserNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientOpenPageInNewTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientOpenPageInNewTab,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_PAGE_IN_NEW_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBrowserNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFocusMenuBar,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserFocusMenuBar,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_MENU_BAR,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBrowserNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFocusBookmarks,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserFocusBookmarks,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_BOOKMARKS,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBrowserNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserNextPane,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserNextPane,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEXT_PANE,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBrowserNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kFocusPreviousPane,
        AcceleratorLayoutDetails(
            AcceleratorAction::kFocusPreviousPane,
            IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_PREVIOUS_PANE,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBrowserNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kBrowserBack,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserBack,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_BACK,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserForward,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserForward,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_FORWARD,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserHome,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserHome,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_HOME,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserReload,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserReload,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_RELOAD,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserReloadBypassingCache,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserReloadBypassingCache,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_RELOAD_BYPASSING_CACHE,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kPages,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserPageUp,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserPageUp,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_PAGE_UP,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserPageDown,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserPageDown,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_PAGE_DOWN,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserTopPage,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserTopPage,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_TOP_PAGE,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserBottomPage,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserBottomPage,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOTTOM_PAGE,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kPages,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserZoomPlus,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserZoomPlus,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_PLUS,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserZoomMinus,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserZoomMinus,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_MINUS,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserZoomNormal,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserZoomNormal,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_NORMAL,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kPages,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFind,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserFind,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserSavePage,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserSavePage,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_SAVE_PAGE,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFindNext,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserFindNext,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND_NEXT,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kPages,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFindPrevious,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserFindPrevious,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND_PREVIOUS,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kPages,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserNewTab,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserNewTab,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_NEW_TAB,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kTabs,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserSelectNextTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserSelectNextTab,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_SELECT_NEXT_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserSelectPreviousTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserSelectPreviousTab,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_SELECT_PREVIOU_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserNewWindow,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserNewWindow,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEW_WINDOW,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kTabs,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserNewIncognitoWindow,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserNewIncognitoWindow,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEW_INCOGNITO_WINDOW,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserTabSearch,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserTabSearch,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_TAB_SEARCH,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kTabs,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserCloseTab,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserCloseTab,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLOSE_TAB,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kTabs,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserRestoreTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserRestoreTab,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_RESTORE_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserSelectLastTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserSelectLastTab,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_SELECT_LAST_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserSelectTabByIndex,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserSelectTabByIndex,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientDragLinkInSameTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientDragLinkInSameTab,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DRAG_LINK_IN_SAME_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientDragLinkInNewTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientDragLinkInNewTab,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DRAG_LINK_IN_NEW_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientOpenLinkInWindow,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientOpenLinkInWindow,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_LINK_IN_WINDOW,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientOpenLinkInTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientOpenLinkInTab,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_LINK_IN_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientOpenLinkInTabBackground,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientOpenLinkInTabBackground,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_LINK_IN_TAB_BACKGROUND,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserStopDragTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserStopDragTab,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_STOP_DRAG_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kTabs,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserBookmarkThisTab,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserBookmarkThisTab,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOOKMARK_THIS_TAB,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBookmarks,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientSaveLinkAsBookmark,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientSaveLinkAsBookmark,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SAVE_LINK_AS_BOOKMARK,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBookmarks,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserBookmarkAllTabs,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserBookmarkAllTabs,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOOKMARK_ALL_TABS,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBookmarks,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserShowBookmarkBar,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserShowBookmarkBar,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_BOOKMARK_BAR,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBookmarks,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserShowBookmarkManager,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserShowBookmarkManager,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_BOOKMARK_MANAGER,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kBookmarks,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserDevToolsConsole,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserDevToolsConsole,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS_CONSOLE,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kDeveloperTools,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserDevToolsInspect,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserDevToolsInspect,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS_INSPECT,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kDeveloperTools,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserDevTools,
        AcceleratorLayoutDetails(NonConfigurableActions::kBrowserDevTools,
                                 IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS,
                                 mojom::AcceleratorCategory::kBrowser,
                                 mojom::AcceleratorSubcategory::kDeveloperTools,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserViewSource,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserViewSource,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_VIEW_SOURCE,
            mojom::AcceleratorCategory::kBrowser,
            mojom::AcceleratorSubcategory::kDeveloperTools,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},

       // Text
       {NonConfigurableActions::kAmbientGoToBeginningOfDocument,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientGoToBeginningOfDocument,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_BEGINNING_OF_DOCUMENT,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientGoToEndOfDocument,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientGoToEndOfDocument,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_END_OF_DOCUMENT,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientGoToBeginningOfLine,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientGoToBeginningOfLine,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_BEGINNING_OF_LINE,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientGoToEndOfLine,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientGoToEndOfLine,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_END_OF_LINE,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientMoveStartOfPreviousWord,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientMoveStartOfPreviousWord,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_MOVE_TO_START_OF_PREVIOUS_WORD,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientMoveToEndOfWord,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientMoveToEndOfWord,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_MOVE_TO_END_OF_WORD,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kToggleCapsLock,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleCapsLock,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_CAPS_LOCK,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kShowEmojiPicker,
        AcceleratorLayoutDetails(
            AcceleratorAction::kShowEmojiPicker,
            IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_EMOJI_PICKER,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kAmbientCopy,
        AcceleratorLayoutDetails(NonConfigurableActions::kAmbientCopy,
                                 IDS_AMBIENT_ACCELERATOR_DESCRIPTION_COPY,
                                 mojom::AcceleratorCategory::kText,
                                 mojom::AcceleratorSubcategory::kTextEditing,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientCut,
        AcceleratorLayoutDetails(NonConfigurableActions::kAmbientCut,
                                 IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CUT,
                                 mojom::AcceleratorCategory::kText,
                                 mojom::AcceleratorSubcategory::kTextEditing,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientPaste,
        AcceleratorLayoutDetails(NonConfigurableActions::kAmbientPaste,
                                 IDS_AMBIENT_ACCELERATOR_DESCRIPTION_PASTE,
                                 mojom::AcceleratorCategory::kText,
                                 mojom::AcceleratorSubcategory::kTextEditing,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientPastePlainText,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientPastePlainText,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_PASTE_PLAIN_TEXT,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kToggleClipboardHistory,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleClipboardHistory,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_CLIPBOARD_HISTORY,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kAmbientDeletePreviousWord,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientDeletePreviousWord,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DELETE_PREVIOUS_WORD,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientUndo,
        AcceleratorLayoutDetails(NonConfigurableActions::kAmbientUndo,
                                 IDS_AMBIENT_ACCELERATOR_DESCRIPTION_UNDO,
                                 mojom::AcceleratorCategory::kText,
                                 mojom::AcceleratorSubcategory::kTextEditing,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientRedo,
        AcceleratorLayoutDetails(NonConfigurableActions::kAmbientRedo,
                                 IDS_AMBIENT_ACCELERATOR_DESCRIPTION_REDO,
                                 mojom::AcceleratorCategory::kText,
                                 mojom::AcceleratorSubcategory::kTextEditing,
                                 /*locked=*/true,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientContentContextSelectAll,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientContentContextSelectAll,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CONTENT_CONTEXT_SELECTALL,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientSelectTextToBeginning,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientSelectTextToBeginning,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SELECT_TEXT_TO_BEGINNING,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientSelectTextToEndOfLine,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientSelectTextToEndOfLine,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SELECT_TEXT_TO_END_OF_LINE,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientSelectPreviousWord,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientSelectPreviousWord,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SELECT_PREVIOUS_WORD,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAMbientSelectNextWord,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAMbientSelectNextWord,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SELECT_NEXT_WORD,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientDeleteNextWord,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientDeleteNextWord,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DELETE_NEXT_WORD,
            mojom::AcceleratorCategory::kText,
            mojom::AcceleratorSubcategory::kTextEditing,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},

       // Windows and Desks
       {NonConfigurableActions::kAmbientCycleForwardMRU,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientCycleForwardMRU,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CYCLE_FORWARD_MRU,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientCycleBackwardMRU,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientCycleBackwardMRU,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CYCLE_BACKWARD_MRU,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kToggleMaximized,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleMaximized,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MAXIMIZED,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kWindowMinimize,
        AcceleratorLayoutDetails(
            AcceleratorAction::kWindowMinimize,
            IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_MINIMIZE,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleFullscreen,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleFullscreen,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_FULLSCREEN,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kBrowserCloseWindow,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserCloseWindow,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLOSE_WINDOW,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kToggleMultitaskMenu,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleMultitaskMenu,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MULTITASK_MENU,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kWindowCycleSnapLeft,
        AcceleratorLayoutDetails(
            AcceleratorAction::kWindowCycleSnapLeft,
            IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_CYCLE_SNAP_LEFT,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kWindowCycleSnapRight,
        AcceleratorLayoutDetails(
            AcceleratorAction::kWindowCycleSnapRight,
            IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_CYCLE_SNAP_RIGHT,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMoveActiveWindowBetweenDisplays,
        AcceleratorLayoutDetails(
            AcceleratorAction::kMoveActiveWindowBetweenDisplays,
            IDS_ASH_ACCELERATOR_DESCRIPTION_MOVE_ACTIVE_WINDOW_BETWEEN_DISPLAYS,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMinimizeTopWindowOnBack,
        AcceleratorLayoutDetails(
            AcceleratorAction::kMinimizeTopWindowOnBack,
            IDS_ASH_ACCELERATOR_DESCRIPTION_MINIMIZE_TOP_WINDOW_ON_BACK,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleSnapGroupWindowsGroupAndUngroup,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleSnapGroupWindowsGroupAndUngroup,
            IDS_ASH_ACCELERATOR_DESCRIPTION_GROUP_UNGROUP_WINDOWS,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleSnapGroupWindowsMinimizeAndRestore,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleSnapGroupWindowsMinimizeAndRestore,
            IDS_ASH_ACCELERATOR_DESCRIPTION_MINIMIZE_RESTORE_WINDOWS,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleFloating,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleFloating,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_FLOATING,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kWindows,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kDesksNewDesk,
        AcceleratorLayoutDetails(AcceleratorAction::kDesksNewDesk,
                                 IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_NEW_DESK,
                                 mojom::AcceleratorCategory::kWindowsAndDesks,
                                 mojom::AcceleratorSubcategory::kDesks,
                                 /*locked=*/false,
                                 mojom::AcceleratorLayoutStyle::kDefault,
                                 mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kDesksRemoveCurrentDesk,
        AcceleratorLayoutDetails(
            AcceleratorAction::kDesksRemoveCurrentDesk,
            IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_REMOVE_CURRENT_DESK,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kDesks,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kDesksActivateDeskLeft,
        AcceleratorLayoutDetails(
            AcceleratorAction::kDesksActivateDeskLeft,
            IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_ACTIVATE_DESK_LEFT,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kDesks,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kDesksActivateDeskRight,
        AcceleratorLayoutDetails(
            AcceleratorAction::kDesksActivateDeskRight,
            IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_ACTIVATE_DESK_RIGHT,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kDesks,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kDesksMoveActiveItemLeft,
        AcceleratorLayoutDetails(
            AcceleratorAction::kDesksMoveActiveItemLeft,
            IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_MOVE_ACTIVE_ITEM_LEFT,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kDesks,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kDesksMoveActiveItemRight,
        AcceleratorLayoutDetails(
            AcceleratorAction::kDesksMoveActiveItemRight,
            IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_MOVE_ACTIVE_ITEM_RIGHT,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kDesks,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kAmbientActivateIndexedDesk,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientActivateIndexedDesk,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kDesks,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kDesksToggleAssignToAllDesks,
        AcceleratorLayoutDetails(
            AcceleratorAction::kDesksToggleAssignToAllDesks,
            IDS_ASH_ACCELERATOR_ACTIONDESKS_TOGGLE_ASSIGN_TO_ALL_DESKS,
            mojom::AcceleratorCategory::kWindowsAndDesks,
            mojom::AcceleratorSubcategory::kDesks,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},

       // Accessibility
       // TODO(jimmyxgong): Allow this to be modifiable but only after revising
       // the notification that hardcodes ctrl + alt + z into the notification
       // message.
       {AcceleratorAction::kToggleSpokenFeedback,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleSpokenFeedback,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_SPOKEN_FEEDBACK,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kChromeVox,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleHighContrast,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleHighContrast,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_HIGH_CONTRAST,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kVisibility,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleDockedMagnifier,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleDockedMagnifier,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_DOCKED_MAGNIFIER,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kVisibility,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kToggleFullscreenMagnifier,
        AcceleratorLayoutDetails(
            AcceleratorAction::kToggleFullscreenMagnifier,
            IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_FULLSCREEN_MAGNIFIER,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kVisibility,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMagnifierZoomIn,
        AcceleratorLayoutDetails(
            AcceleratorAction::kMagnifierZoomIn,
            IDS_ASH_ACCELERATOR_DESCRIPTION_MAGNIFIER_ZOOM_IN,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kVisibility,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {AcceleratorAction::kMagnifierZoomOut,
        AcceleratorLayoutDetails(
            AcceleratorAction::kMagnifierZoomOut,
            IDS_ASH_ACCELERATOR_DESCRIPTION_MAGNIFIER_ZOOM_OUT,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kVisibility,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kAmbientSwitchFocusForwards,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientSwitchFocusForwards,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SWITCH_FOCUS_FORWARDS,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientSwitchFocusBackwards,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientSwitchFocusBackwards,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SWITCH_FOCUS_BACKWARDS,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientCaretBrowsing,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientCaretBrowsing,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CARET_BROWSING,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {AcceleratorAction::kFocusShelf,
        AcceleratorLayoutDetails(
            AcceleratorAction::kFocusShelf,
            IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_SHELF,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAsh)},
       {NonConfigurableActions::kAmbientHighlightNextItemOnShelf,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientHighlightNextItemOnShelf,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_HIGHLIGHT_NEXT_ITEM_ON_SHELF,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientHighlightPreviousItemOnShelf,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientHighlightPreviousItemOnShelf,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_HIGHTLIGHT_PREVIOUS_ITEM_ON_SHELF,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientOpenHighlightedItemOnShelf,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientOpenHighlightedItemOnShelf,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_HIGHLIGHTED_ITEM_ON_SHELF,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientRemoveHighlightOnShelf,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientRemoveHighlightOnShelf,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_REMOVE_HIGHLIGHT_ON_SHELF,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientOpenRightClickMenu,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientOpenRightClickMenu,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_RIGHT_CLICK_MENU,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFocusInactivePopupForAccessibility,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserFocusInactivePopupForAccessibility,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_INACTIVE_POPUP_FOR_ACCESSIBILITY,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kBrowserFocusToolbar,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kBrowserFocusToolbar,
            IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_TOOLBAR,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientMoveAppsInGrid,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientMoveAppsInGrid,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_MOVE_APPS_IN_GRID,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)},
       {NonConfigurableActions::kAmbientMoveAppsInOutFolder,
        AcceleratorLayoutDetails(
            NonConfigurableActions::kAmbientMoveAppsInOutFolder,
            IDS_AMBIENT_ACCELERATOR_DESCRIPTION_MOVE_APPS_IN_OUT_FOLDER,
            mojom::AcceleratorCategory::kAccessibility,
            mojom::AcceleratorSubcategory::kAccessibilityNavigation,
            /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
            mojom::AcceleratorSource::kAmbient)}});

  return *acceleratorLayoutMap;
}

// The following map are accelerators that will not appear in the app and cannot
// be used as a custom accelerator. For example, if you have an accelerator
// that has a complex text-based instruction that uses a particular accelerator
// this list is useful to reserve those keys.
const ReservedAcceleratorsMap& GetReservedAcceleratorsMap() {
  static base::NoDestructor<ReservedAcceleratorsMap> reservedAcceleratorsMap({
      // NonConfigurableActions::kAmbientCycleForwardMRU.
      {ui::Accelerator(ui::VKEY_TAB, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CYCLE_FORWARD_MRU},
      // NonConfigurableActions::kAmbientCycleBackwardMRU.
      {ui::Accelerator(ui::VKEY_TAB, ui::EF_ALT_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CYCLE_BACKWARD_MRU},
      // The following are already included in the app as
      // `NonConfigurableActions::kAmbientLaunchNumberedApp1.
      {ui::Accelerator(ui::VKEY_1, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP},
      {ui::Accelerator(ui::VKEY_2, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP},
      {ui::Accelerator(ui::VKEY_3, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP},
      {ui::Accelerator(ui::VKEY_4, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP},
      {ui::Accelerator(ui::VKEY_5, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP},
      {ui::Accelerator(ui::VKEY_6, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP},
      {ui::Accelerator(ui::VKEY_7, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP},
      {ui::Accelerator(ui::VKEY_8, ui::EF_ALT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP},
      // The following are already included in the app as
      // `NonConfigurableActions::kBrowserSelectTabByIndex`.
      {ui::Accelerator(ui::VKEY_1, ui::EF_CONTROL_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE},
      {ui::Accelerator(ui::VKEY_2, ui::EF_CONTROL_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE},
      {ui::Accelerator(ui::VKEY_3, ui::EF_CONTROL_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE},
      {ui::Accelerator(ui::VKEY_4, ui::EF_CONTROL_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE},
      {ui::Accelerator(ui::VKEY_5, ui::EF_CONTROL_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE},
      {ui::Accelerator(ui::VKEY_6, ui::EF_CONTROL_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE},
      {ui::Accelerator(ui::VKEY_7, ui::EF_CONTROL_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE},
      {ui::Accelerator(ui::VKEY_8, ui::EF_CONTROL_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE},
      // The following are already included in the app as
      // `NonConfigurableActions::kAmbientActivateIndexedDesk`.
      {ui::Accelerator(ui::VKEY_1, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK},
      {ui::Accelerator(ui::VKEY_2, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK},
      {ui::Accelerator(ui::VKEY_3, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK},
      {ui::Accelerator(ui::VKEY_4, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK},
      {ui::Accelerator(ui::VKEY_5, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK},
      {ui::Accelerator(ui::VKEY_6, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK},
      {ui::Accelerator(ui::VKEY_7, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK},
      {ui::Accelerator(ui::VKEY_8, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK},
      {ui::Accelerator(ui::VKEY_ESCAPE, ui::EF_COMMAND_DOWN | ui::EF_SHIFT_DOWN,
                       ui::Accelerator::KeyState::PRESSED),
       IDS_ASH_ACCELERATOR_DESCRIPTION_UNPIN},
  });

  return *reservedAcceleratorsMap;
}

std::optional<AcceleratorLayoutDetails> GetAcceleratorLayout(uint32_t id) {
  const auto& layout_iter = GetAcceleratorLayoutMap().find(id);
  if (layout_iter == GetAcceleratorLayoutMap().end()) {
    return std::nullopt;
  }
  return layout_iter->second;
}

}  // namespace ash
