/* SPDX-License-Identifier: BSD-3-Clause
 * Copyright(c) 2018 Cavium Networks
 * Copyright (c) 2019 Intel Corporation
 */

#ifndef TEST_CRYPTODEV_MOD_TEST_VECTORS_H_
#define TEST_CRYPTODEV_MOD_TEST_VECTORS_H_

#define DATA_SIZE 512

struct modex_test_data {
	enum rte_crypto_asym_xform_type xform_type;
	const char *description;
	struct {
		uint8_t data[DATA_SIZE];
		uint16_t len;
	} base;
	struct {
		uint8_t data[DATA_SIZE];
		uint16_t len;
	} exponent;
	struct {
		uint8_t data[DATA_SIZE];
		uint16_t len;
	} modulus;
	struct {
		uint8_t data[DATA_SIZE];
		uint16_t len;
	} reminder;
	uint16_t result_len;
};
struct modinv_test_data {
	enum rte_crypto_asym_xform_type xform_type;
	const char *description;
	struct {
		uint8_t data[DATA_SIZE];
		uint16_t len;
	} base;
	struct {
		uint8_t data[DATA_SIZE];
		uint16_t len;
	} modulus;
	struct {
		uint8_t data[DATA_SIZE];
		uint16_t len;
	} inverse;
	uint16_t result_len;
};

static const struct
modex_test_data modex_test_case[] = {
{
	.description = "Modular Exponentiation "
				   "(mod=128, base=20, exp=3, res=128)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0xF8, 0xBA, 0x1A, 0x55, 0xD0, 0x2F, 0x85,
			0xAE, 0x96, 0x7B, 0xB6, 0x2F, 0xB6, 0xCD,
			0xA8, 0xEB, 0x7E, 0x78, 0xA0, 0x50
		},
		.len = 20
	},
	.exponent = {
		.data = {
			0x01, 0x00, 0x01
		},
		.len = 3
	},
	.reminder = {
		.data = {
			0x2C, 0x60, 0x75, 0x45, 0x98, 0x9D, 0xE0, 0x72,
			0xA0, 0x9D, 0x3A, 0x9E, 0x03, 0x38, 0x73, 0x3C,
			0x31, 0x83, 0x04, 0xFE, 0x75, 0x43, 0xE6, 0x17,
			0x5C, 0x01, 0x29, 0x51, 0x69, 0x33, 0x62, 0x2D,
			0x78, 0xBE, 0xAE, 0xC4, 0xBC, 0xDE, 0x7E, 0x2C,
			0x77, 0x84, 0xF2, 0xC5, 0x14, 0xB5, 0x2F, 0xF7,
			0xC5, 0x94, 0xEF, 0x86, 0x75, 0x75, 0xB5, 0x11,
			0xE5, 0x0E, 0x0A, 0x29, 0x76, 0xE2, 0xEA, 0x32,
			0x0E, 0x43, 0x77, 0x7E, 0x2C, 0x27, 0xAC, 0x3B,
			0x86, 0xA5, 0xDB, 0xC9, 0x48, 0x40, 0xE8, 0x99,
			0x9A, 0x0A, 0x3D, 0xD6, 0x74, 0xFA, 0x2E, 0x2E,
			0x5B, 0xAF, 0x8C, 0x99, 0x44, 0x2A, 0x67, 0x38,
			0x27, 0x41, 0x59, 0x9D, 0xB8, 0x51, 0xC9, 0xF7,
			0x43, 0x61, 0x31, 0x6E, 0xF1, 0x25, 0x38, 0x7F,
			0xAE, 0xC6, 0xD0, 0xBB, 0x29, 0x76, 0x3F, 0x46,
			0x2E, 0x1B, 0xE4, 0x67, 0x71, 0xE3, 0x87, 0x5A
		},
		.len = 128
	},
	.modulus = {
		.data = {
			0xb3, 0xa1, 0xaf, 0xb7, 0x13, 0x08, 0x00, 0x0a,
			0x35, 0xdc, 0x2b, 0x20, 0x8d, 0xa1, 0xb5, 0xce,
			0x47, 0x8a, 0xc3, 0x80, 0xf4, 0x7d, 0x4a, 0xa2,
			0x62, 0xfd, 0x61, 0x7f, 0xb5, 0xa8, 0xde, 0x0a,
			0x17, 0x97, 0xa0, 0xbf, 0xdf, 0x56, 0x5a, 0x3d,
			0x51, 0x56, 0x4f, 0x70, 0x70, 0x3f, 0x63, 0x6a,
			0x44, 0x5b, 0xad, 0x84, 0x0d, 0x3f, 0x27, 0x6e,
			0x3b, 0x34, 0x91, 0x60, 0x14, 0xb9, 0xaa, 0x72,
			0xfd, 0xa3, 0x64, 0xd2, 0x03, 0xa7, 0x53, 0x87,
			0x9e, 0x88, 0x0b, 0xc1, 0x14, 0x93, 0x1a, 0x62,
			0xff, 0xb1, 0x5d, 0x74, 0xcd, 0x59, 0x63, 0x18,
			0x11, 0x3d, 0x4f, 0xba, 0x75, 0xd4, 0x33, 0x4e,
			0x23, 0x6b, 0x7b, 0x57, 0x44, 0xe1, 0xd3, 0x03,
			0x13, 0xa6, 0xf0, 0x8b, 0x60, 0xb0, 0x9e, 0xee,
			0x75, 0x08, 0x9d, 0x71, 0x63, 0x13, 0xcb, 0xa6,
			0x81, 0x92, 0x14, 0x03, 0x22, 0x2d, 0xde, 0x55
		},
		.len = 128
	},
	.result_len = 128
},
{
	.description = "Modular Exponentiation "
				   "(mod=60, base=50, exp=40, res=60)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0x4F, 0xD8, 0x5C, 0xDB, 0x6D, 0xA2, 0xFA, 0x35,
			0x9D, 0xD7, 0x97, 0x10, 0x4B, 0x71, 0x5F, 0x53,
			0xE1, 0xC7, 0x09, 0x74, 0x88, 0xC8, 0x9D, 0x03,
			0xC0, 0x70, 0xE0, 0xBE, 0xE3, 0xF2, 0x2C, 0x01,
			0x85, 0xA6, 0x4E, 0x28, 0x6E, 0xD3, 0xB5, 0x18,
			0x58, 0x69, 0x07, 0xDA, 0x3A, 0x1B, 0x35, 0xCE,
			0xE6, 0xFA
		},
		.len = 50
	},
	.exponent = {
		.data = {
			0x30, 0xA5, 0xD5, 0xF0, 0x42, 0x03, 0xC3, 0x2D,
			0x2F, 0x58, 0xA8, 0x5C, 0x21, 0x88, 0xDE, 0x82,
			0x36, 0x44, 0xC1, 0x5A, 0x87, 0x2C, 0x33, 0x19,
			0x4E, 0xCE, 0x3F, 0x87, 0xFF, 0x98, 0x4B, 0xFC,
			0x15, 0xC0, 0xBE, 0x9E, 0x8F, 0xF0, 0x6A, 0x62
		},
		.len = 40
	},
	.reminder = {
		.data = {
			0x59, 0x3D, 0x92, 0xE4, 0xE6, 0x9B, 0x5D, 0x97,
			0x21, 0xE6, 0x06, 0x60, 0x8B, 0x66, 0x8F, 0xF3,
			0x63, 0xAE, 0x3A, 0x64, 0x7F, 0xCA, 0xDE, 0x4A,
			0xE5, 0x49, 0xA2, 0x90, 0x0F, 0xAE, 0x13, 0x10,
			0xC2, 0x5B, 0x97, 0xED, 0x86, 0x97, 0x16, 0xE2,
			0x01, 0xF6, 0x7A, 0xF1, 0x7B, 0x4B, 0xC7, 0x7D,
			0x6B, 0xD7, 0x98, 0xE5, 0x9B, 0x75, 0x1A, 0x6B,
			0xD6, 0x6E, 0xA3, 0x4E
		},
		.len = 60
	},
	.modulus = {
		.data = {
			0xF8, 0x04, 0x0D, 0xD5, 0x09, 0x6C, 0x78, 0x06,
			0x7D, 0x28, 0x77, 0xA4, 0x0E, 0xA5, 0x49, 0xE7,
			0x6D, 0xC9, 0x97, 0xD3, 0xC0, 0x7F, 0x82, 0xC6,
			0x75, 0x51, 0x72, 0xAF, 0x8C, 0x77, 0x97, 0xD0,
			0xA1, 0x85, 0x54, 0xC0, 0x78, 0x86, 0xD6, 0x40,
			0x7A, 0x6B, 0xB3, 0xD7, 0x07, 0xCA, 0x27, 0xA3,
			0x66, 0xB9, 0x98, 0x22, 0xC4, 0x54, 0x18, 0x07,
			0x65, 0x76, 0x0F, 0x5A
		},
		.len = 60
	},
	.result_len = 60
},
{
	.description = "Modular Exponentiation "
				   "(mod=8, base=65, exp=17, res=8)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0x25, 0x74, 0x41, 0xCE, 0xFA, 0x5C, 0x07, 0x2A,
			0xD1, 0x74, 0xF3, 0x3D, 0xE1, 0xCC, 0xC3, 0x18,
			0x7E, 0x4A, 0x21, 0x9F, 0x97, 0xA3, 0x26, 0x85,
			0x85, 0xD9, 0x9B, 0xE3, 0xBA, 0xB3, 0x70, 0xC9,
			0x26, 0x68, 0xE4, 0xB7, 0x4C, 0x88, 0x48, 0xC1,
			0x6B, 0xC6, 0x3C, 0x00, 0x8C, 0x6B, 0xC6, 0x11,
			0xD0, 0xD6, 0x61, 0x5D, 0xEC, 0xAA, 0xBA, 0x3B,
			0x7D, 0xB3, 0x0D, 0x3F, 0xA5, 0x4D, 0xEE, 0xE4,
			0xAC
		},
		.len = 65
	},
	.exponent = {
		.data = {
			0x1C, 0x54, 0x2F, 0xCA, 0xDE, 0x4F, 0x17, 0x38,
			0x69, 0x87, 0xB4, 0xFF, 0x3A, 0x6C, 0x82, 0x70,
			0x53
		},
		.len = 17
	},
	.reminder = {
		.data = {
			0x52, 0x06, 0x1A, 0x35, 0x70, 0x33, 0x78, 0x45
		},
		.len = 8
	},
	.modulus = {
		.data = {
			0x6B, 0x6D, 0xFA, 0xCB, 0x09, 0x5D, 0x9C, 0xFD
		},
		.len = 8
	},
	.result_len = 8
},
{
	.description = "Modular Exponentiation "
				   "(mod=100, base=150, exp=192, res=100)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0xC1, 0xA1, 0x04, 0xE4, 0x4D, 0x4A, 0xD3, 0x5C,
			0xB3, 0xD0, 0x16, 0x51, 0xA7, 0xF4, 0x82, 0x6C,
			0x22, 0xDD, 0x4D, 0xAA, 0x70, 0x30, 0x25, 0xA7,
			0xFA, 0xA9, 0xF2, 0x20, 0x55, 0x9B, 0xEA, 0x26,
			0xF6, 0xB5, 0xF8, 0x9C, 0x46, 0x21, 0x85, 0x0E,
			0x38, 0x73, 0x01, 0xC6, 0x72, 0x67, 0x9B, 0x49,
			0xCE, 0x48, 0xB7, 0x4A, 0xEE, 0x08, 0x21, 0x26,
			0xF3, 0x21, 0x77, 0xE7, 0x3C, 0x77, 0xF4, 0x0A,
			0x82, 0xC8, 0x16, 0x94, 0x6C, 0xBF, 0xA8, 0xD8,
			0x8B, 0x7D, 0x27, 0x60, 0xAC, 0x06, 0x69, 0x7E,
			0x46, 0x2C, 0xE2, 0xD1, 0x13, 0x50, 0x7D, 0xCE,
			0x4D, 0xC4, 0x5D, 0x81, 0xFB, 0x6B, 0x11, 0x4A,
			0x2D, 0xA2, 0x03, 0x55, 0x77, 0x8C, 0x3D, 0xA1,
			0xAD, 0xBE, 0x9C, 0x72, 0xE5, 0xA5, 0xFB, 0x49,
			0x5F, 0x13, 0x48, 0xC7, 0xAC, 0xD1, 0x0F, 0x5E,
			0xDF, 0x9C, 0xC7, 0xF5, 0x19, 0xFD, 0xC5, 0x77,
			0x27, 0x8D, 0xC4, 0x1D, 0x90, 0x8C, 0x20, 0x96,
			0xC8, 0x6A, 0x0D, 0x2F, 0xE2, 0x8B, 0xB0, 0x58,
			0xF8, 0xC4, 0x31, 0x0A, 0x17, 0x11
		},
		.len = 150
	},
	.exponent = {
		.data = {
			0xC6, 0x20, 0x99, 0xD9, 0xBC, 0xE2, 0xAD, 0x74,
			0x11, 0x6F, 0x74, 0x14, 0x72, 0xB8, 0x09, 0xCB,
			0x5C, 0x74, 0x11, 0x21, 0x17, 0x84, 0x02, 0xDC,
			0x70, 0x59, 0x20, 0x79, 0x40, 0x7B, 0x0E, 0x52,
			0xAD, 0x00, 0x38, 0x4F, 0x5A, 0xE5, 0x0D, 0x28,
			0xB5, 0xF8, 0xDC, 0x54, 0x92, 0xB2, 0xB0, 0xA8,
			0xE8, 0x35, 0x1B, 0x63, 0x0D, 0x6A, 0x50, 0x8D,
			0xE1, 0x3E, 0x7A, 0xDD, 0x42, 0x7A, 0xD0, 0xB4,
			0x9D, 0x63, 0x36, 0x03, 0xC0, 0x9B, 0xA0, 0x91,
			0x8B, 0xBC, 0x45, 0x53, 0x93, 0x2C, 0xFC, 0xDD,
			0x4F, 0xBD, 0x96, 0x0B, 0x63, 0xEB, 0xEF, 0x50,
			0xAC, 0x99, 0x45, 0xA7, 0x0D, 0xC6, 0xEA, 0x98,
			0xBC, 0xD7, 0x63, 0x56, 0x8C, 0x75, 0x68, 0xAE,
			0xF0, 0xB4, 0x66, 0xA0, 0x4D, 0xC5, 0x71, 0xB9,
			0x4E, 0xCB, 0xF6, 0xCA, 0xC9, 0x1B, 0x3B, 0x55,
			0x91, 0x39, 0x25, 0xBD, 0x98, 0xAA, 0xDA, 0xF2,
			0x8A, 0xCB, 0x8E, 0x56, 0x09, 0xBF, 0xC4, 0x1D,
			0xFA, 0x23, 0x48, 0xF6, 0x9A, 0xD2, 0xD3, 0x2B,
			0xED, 0x60, 0x9B, 0x4B, 0x63, 0xD8, 0x8C, 0x6A,
			0x28, 0xA3, 0x4C, 0x85, 0x43, 0x4D, 0x5C, 0x4A,
			0xA7, 0xA9, 0x9F, 0x7E, 0x13, 0x5B, 0x36, 0xED,
			0xD9, 0x53, 0xBE, 0x12, 0xFF, 0x17, 0x9F, 0x70,
			0xA4, 0xD2, 0x42, 0x72, 0x70, 0x51, 0x70, 0x3F,
			0x5A, 0xBA, 0x33, 0x0E, 0xBB, 0x4C, 0xA0, 0x4A
		},
		.len = 192
	},
	.reminder = {
		.data = {
			0x07, 0x5E, 0x28, 0x4F, 0xD1, 0xEA, 0x5C, 0x1D,
			0xF8, 0xBF, 0x29, 0xF0, 0x63, 0xCC, 0xF7, 0x6D,
			0x99, 0x67, 0xCE, 0xE3, 0x05, 0x16, 0x16, 0x8C,
			0x3A, 0x07, 0xC0, 0x63, 0x70, 0xB9, 0x1A, 0x24,
			0xED, 0xE9, 0xF0, 0xEE, 0xD9, 0xAB, 0x18, 0xD4,
			0x59, 0xB4, 0xD2, 0x77, 0x44, 0x94, 0x72, 0xFE,
			0x19, 0x26, 0x50, 0x47, 0x77, 0xAD, 0x0A, 0x45,
			0x76, 0x4B, 0x22, 0xDB, 0x05, 0x13, 0x67, 0x40,
			0x9A, 0x36, 0x6C, 0x5E, 0xE8, 0xED, 0x40, 0x60,
			0x86, 0x40, 0x2F, 0x30, 0x9E, 0x4B, 0x61, 0x73,
			0x2E, 0x76, 0x8A, 0xB0, 0x49, 0x04, 0x1A, 0x82,
			0xB7, 0xEF, 0xB2, 0xB5, 0xB4, 0xE0, 0x87, 0xF0,
			0xB4, 0x53, 0xB2, 0xBE
		},
		.len = 100
	},
	.modulus = {
		.data = {
			0x54, 0x58, 0x5C, 0xBA, 0xAE, 0xC1, 0xB4, 0x46,
			0x50, 0xAF, 0xD0, 0xA6, 0x03, 0x9D, 0x74, 0x84,
			0x6F, 0x89, 0x07, 0xA6, 0x63, 0xE7, 0x34, 0xB2,
			0x55, 0x0E, 0xD5, 0x42, 0xC9, 0xBF, 0xD1, 0x89,
			0x54, 0x0B, 0x76, 0xF7, 0x0E, 0xA1, 0x42, 0x02,
			0x72, 0xDC, 0x28, 0x5A, 0x68, 0x10, 0xA0, 0x84,
			0xA4, 0x72, 0x4D, 0x40, 0x69, 0xBC, 0x18, 0xC9,
			0x92, 0x69, 0xB8, 0x52, 0x2A, 0xB1, 0xA3, 0x43,
			0x80, 0xA9, 0x55, 0x78, 0xEA, 0xD9, 0x54, 0xF4,
			0x3A, 0xDD, 0x24, 0x4E, 0x22, 0x9D, 0x89, 0x40,
			0x8F, 0x50, 0xA5, 0xF5, 0x0F, 0xFA, 0x38, 0xBB,
			0xE8, 0xD7, 0x21, 0x6B, 0xEA, 0xB1, 0x28, 0x48,
			0xEB, 0x75, 0xB1, 0xC7
		},
		.len = 100
	},
	.result_len = 100
},
{
	.description = "Modular Exponentiation "
				   "(mod=255, base=20, exp=10, res=255)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0xD8, 0x21, 0xD2, 0x76, 0xAE, 0x01, 0x62, 0xD8,
			0x8C, 0x80, 0x01, 0x25, 0xC7, 0xE8, 0x4E, 0x0F,
			0x7F, 0x23, 0xFE, 0xBB
		},
		.len = 20
	},
	.exponent = {
		.data = {
			0xE5, 0xCE, 0x50, 0xE8, 0x97, 0x32, 0xFB, 0x5C,
			0xFC, 0x62
		},
		.len = 10
	},
	.reminder = {
		.data = {
			0x1F, 0xA8, 0xAC, 0x32, 0x6C, 0x20, 0xBE, 0x1D,
			0x88, 0xE9, 0x94, 0x7E, 0xDE, 0xB5, 0x15, 0xA5,
			0xF3, 0x45, 0x77, 0xB3, 0x3E, 0xDD, 0xDD, 0xAE,
			0xEA, 0xAE, 0x32, 0x81, 0x37, 0xAE, 0x26, 0xC5,
			0x6F, 0x3A, 0x75, 0x1F, 0x4F, 0xA2, 0xA3, 0x8B,
			0xDC, 0x52, 0x3C, 0xD5, 0x42, 0x00, 0x58, 0x37,
			0xB3, 0x5B, 0xC9, 0x78, 0xFA, 0x48, 0xFE, 0x4B,
			0x81, 0xFB, 0x4C, 0x5D, 0x9F, 0x5E, 0x4F, 0x56,
			0x30, 0x79, 0x71, 0x1E, 0xC5, 0x8E, 0xC2, 0x2E,
			0x9D, 0x18, 0xD8, 0x34, 0x0E, 0xDB, 0xCB, 0x02,
			0xA7, 0x37, 0x00, 0x44, 0x57, 0x2A, 0x11, 0x85,
			0x81, 0x06, 0x28, 0xB4, 0x95, 0x79, 0xE2, 0x60,
			0x39, 0x66, 0x34, 0x2F, 0x19, 0xD4, 0x42, 0x70,
			0xA8, 0xE9, 0x51, 0xBC, 0xB4, 0x04, 0xD6, 0x4B,
			0xE1, 0x4A, 0xEE, 0x6F, 0x69, 0xD7, 0x82, 0x07,
			0x59, 0xD6, 0xB0, 0xAC, 0x22, 0x28, 0xB8, 0x70,
			0x5F, 0xD2, 0x0D, 0x15, 0x6D, 0xED, 0xEB, 0x8A,
			0xD5, 0x64, 0x16, 0x7C, 0x12, 0x25, 0x76, 0xFD,
			0x8A, 0x8D, 0xC5, 0xEA, 0xF8, 0xA9, 0x04, 0x65,
			0x58, 0x6D, 0x67, 0x2F, 0xA7, 0x3E, 0x2C, 0x96,
			0x55, 0x11, 0xAA, 0x10, 0xE0, 0x1B, 0x0F, 0x1C,
			0x89, 0x10, 0x7C, 0x63, 0xCB, 0x0B, 0xB8, 0x1C,
			0x49, 0x16, 0xC1, 0xB3, 0xA5, 0x57, 0x87, 0x4F,
			0x80, 0x8B, 0x18, 0xF0, 0x46, 0xCE, 0xE4, 0x95,
			0x2F, 0x9C, 0xE7, 0xB7, 0x40, 0x1C, 0xB4, 0xA3,
			0xD4, 0x30, 0x9C, 0xFD, 0x95, 0x35, 0x57, 0x10,
			0x50, 0x40, 0xDA, 0x47, 0x8E, 0x21, 0xAE, 0x05,
			0xF8, 0xA8, 0x72, 0xA5, 0x68, 0xE1, 0xA2, 0xAB,
			0x36, 0x0D, 0x4B, 0xDB, 0x7B, 0xBF, 0x7F, 0x51,
			0x51, 0xC9, 0xBB, 0x3F, 0xF7, 0x76, 0x45, 0x78,
			0xAC, 0x3F, 0x56, 0xF4, 0x27, 0x7E, 0xEF, 0x02,
			0xED, 0x41, 0x15, 0xDB, 0xE4, 0x81, 0x59
		},
		.len = 255
	},
	.modulus = {
		.data = {
			0x71, 0x3C, 0x6C, 0x7A, 0x19, 0x31, 0xF8, 0x94,
			0xC9, 0xAA, 0x25, 0x69, 0xA7, 0xF2, 0x28, 0x70,
			0x84, 0x5D, 0xEC, 0x40, 0xC8, 0xF9, 0xC5, 0x79,
			0xF9, 0x87, 0xD1, 0xA0, 0xC1, 0x5A, 0x06, 0xE4,
			0x65, 0xB8, 0x29, 0x0B, 0x2B, 0xFE, 0x67, 0xF0,
			0x91, 0x96, 0xE1, 0xCD, 0x5A, 0xCE, 0x44, 0xA3,
			0x4F, 0xE8, 0xBE, 0xC6, 0xA3, 0x0A, 0xCB, 0xF5,
			0x7D, 0x8B, 0x9B, 0x2F, 0x4E, 0xC9, 0x54, 0x48,
			0xA4, 0xC2, 0x09, 0xCE, 0xA5, 0x93, 0x1F, 0x43,
			0xC2, 0xCE, 0xFB, 0xBB, 0x69, 0x29, 0x03, 0x74,
			0xD6, 0x25, 0x47, 0x6B, 0xAC, 0x4E, 0x44, 0x8C,
			0x39, 0x2F, 0xB2, 0xDD, 0x15, 0x1B, 0xA3, 0x3D,
			0xA4, 0x0C, 0xFF, 0xCB, 0x05, 0xC2, 0x81, 0x97,
			0x16, 0xE2, 0xAC, 0x8A, 0xF3, 0xED, 0x80, 0xA4,
			0xC5, 0xFC, 0xF5, 0x6C, 0x4B, 0xBB, 0x05, 0x91,
			0xD4, 0x0F, 0xDA, 0x70, 0x7C, 0x9A, 0xA1, 0x63,
			0x15, 0xEE, 0xBB, 0x17, 0xE6, 0x20, 0x50, 0x74,
			0x36, 0x9C, 0xA1, 0x10, 0x29, 0x22, 0xFB, 0x7E,
			0x2A, 0x08, 0xF3, 0x07, 0xEA, 0xCD, 0x2C, 0x50,
			0x18, 0x15, 0x66, 0x87, 0x74, 0x19, 0x11, 0x2B,
			0x77, 0x85, 0xA0, 0x57, 0xA7, 0xEB, 0x6E, 0x15,
			0x15, 0x0D, 0xA4, 0x18, 0x5D, 0x54, 0x13, 0xE3,
			0x33, 0x12, 0x8D, 0xA3, 0xEF, 0x54, 0xE6, 0x1E,
			0xDB, 0x8F, 0x3D, 0x02, 0x3C, 0xCB, 0x34, 0x93,
			0x31, 0x1D, 0x4D, 0x3E, 0x9C, 0x22, 0x04, 0xD1,
			0x19, 0x53, 0x45, 0xE5, 0xBF, 0xF8, 0x70, 0x1A,
			0xEA, 0x52, 0x93, 0x2A, 0x26, 0x8A, 0x1E, 0x47,
			0xCE, 0x83, 0x5B, 0x35, 0x9A, 0xD2, 0x75, 0xC6,
			0xC6, 0x20, 0x84, 0x9F, 0x74, 0x69, 0x69, 0xB8,
			0x29, 0xD8, 0xA4, 0x70, 0x91, 0x42, 0x06, 0x25,
			0x38, 0xCB, 0x42, 0x75, 0x52, 0xEF, 0xB8, 0x64,
			0x4F, 0xC5, 0x7C, 0xC4, 0x09, 0xDB, 0x12
		},
		.len = 255
	},
	.result_len = 255
},
{
	.description = "Modular Exponentiation "
				   "(mod=112, base=257, exp=43, res=112)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0x53, 0x63, 0xB0, 0x3A, 0x30, 0xDE, 0x07, 0xBC,
			0xE4, 0x4B, 0x52, 0x37, 0x9C, 0xD8, 0x5A, 0xCD,
			0x03, 0xE1, 0xEA, 0x6D, 0xDE, 0x4C, 0x19, 0xE6,
			0xA2, 0x0F, 0xAE, 0x56, 0xCD, 0xB2, 0xB6, 0x5B,
			0x31, 0xF5, 0x41, 0x48, 0x8D, 0xA2, 0xC7, 0x0C,
			0x19, 0x32, 0x6D, 0x58, 0x10, 0xD5, 0xF0, 0x64,
			0xF9, 0xF5, 0xD7, 0xFE, 0x37, 0x4A, 0xD8, 0xE3,
			0xBF, 0xDF, 0xCB, 0x45, 0xD5, 0xBF, 0xB9, 0x2A,
			0x60, 0xF8, 0x52, 0xB0, 0xB5, 0x22, 0x76, 0xBD,
			0xD3, 0x0D, 0xD4, 0xE1, 0x42, 0xC4, 0x8C, 0x47,
			0x2D, 0x04, 0x25, 0x1B, 0xFB, 0x21, 0xFD, 0x80,
			0xC0, 0xCE, 0x9D, 0x32, 0x76, 0x8E, 0x18, 0x28,
			0xDC, 0x0F, 0x44, 0x37, 0xF8, 0x61, 0x45, 0x93,
			0xD3, 0x62, 0x21, 0xEE, 0x8B, 0x89, 0x8B, 0xAF,
			0x8B, 0xE9, 0xA5, 0xD2, 0x00, 0xF5, 0xFF, 0xE6,
			0xE0, 0x56, 0x9D, 0x41, 0x13, 0xBC, 0xD6, 0x6E,
			0xC9, 0xE8, 0xE8, 0xC7, 0x61, 0x00, 0x7D, 0x91,
			0x59, 0xAC, 0x6A, 0x24, 0x86, 0x3C, 0x50, 0xFB,
			0x49, 0xC4, 0xB9, 0x41, 0xCD, 0xF0, 0xD9, 0xE7,
			0xE1, 0x54, 0x3F, 0x17, 0x3B, 0xC7, 0x12, 0x20,
			0x6E, 0xC5, 0x80, 0x11, 0xA5, 0x78, 0x72, 0xCA,
			0xBC, 0x90, 0xB7, 0xC5, 0xFF, 0x78, 0xE5, 0x71,
			0x62, 0x4C, 0xCC, 0x6C, 0xEA, 0x76, 0xE3, 0xB6,
			0x00, 0x54, 0x31, 0x72, 0x5A, 0xFE, 0x14, 0xC3,
			0x60, 0x3A, 0x79, 0x97, 0x26, 0x87, 0x69, 0x8D,
			0x44, 0x8E, 0x8B, 0xE0, 0xBC, 0x5C, 0x9F, 0xDE,
			0xD0, 0x90, 0xA2, 0x85, 0xC8, 0x3E, 0x7E, 0xA0,
			0x42, 0xE2, 0x3B, 0xEE, 0x0C, 0x59, 0x1E, 0x72,
			0x62, 0xA5, 0xEE, 0x20, 0xE0, 0xFE, 0x0D, 0xD3,
			0x9F, 0xA9, 0x84, 0xBC, 0xD0, 0x6E, 0x5E, 0xC2,
			0x0B, 0xF2, 0xAE, 0xB6, 0xE6, 0xC6, 0x88, 0xF9,
			0x51, 0xF8, 0x02, 0x08, 0xC6, 0x99, 0x73, 0xF2,
			0x36
		},
		.len = 257
	},
	.exponent = {
		.data = {
			0xCA, 0x5C, 0x73, 0xF7, 0x8B, 0x1F, 0x95, 0xE4,
			0x0E, 0x9B, 0x47, 0xDC, 0x03, 0x96, 0x75, 0xB4,
			0x48, 0x74, 0x73, 0xBE, 0xF8, 0x92, 0x80, 0xE4,
			0x93, 0x5D, 0x87, 0x7D, 0x74, 0xF7, 0x45, 0xEF,
			0x8E, 0x53, 0x9C, 0x03, 0xB0, 0xD6, 0xF3, 0xBF,
			0x86, 0xB2, 0xCD
		},
		.len = 43
	},
	.reminder = {
		.data = {
			0x01, 0x0E, 0x8C, 0x1B, 0x19, 0xF6, 0xB0, 0x0D,
			0x8D, 0xFF, 0x12, 0x74, 0xF0, 0xD0, 0xB1, 0xA2,
			0x49, 0xA4, 0xA3, 0x9C, 0x4D, 0xFA, 0xA7, 0xB7,
			0x6E, 0x45, 0xCC, 0x0D, 0x75, 0xC7, 0xF7, 0x99,
			0x1F, 0x01, 0x44, 0x7B, 0xF2, 0xF9, 0x73, 0x67,
			0x75, 0xD9, 0x4C, 0x2F, 0xA9, 0xB4, 0x59, 0x9E,
			0xF9, 0x2C, 0xB9, 0x14, 0x5D, 0x5C, 0x18, 0x72,
			0xEC, 0x27, 0x1A, 0x2D, 0xFB, 0xDA, 0xEB, 0x2F,
			0x98, 0xA9, 0xC2, 0x01, 0x75, 0x7A, 0x27, 0x07,
			0x94, 0x71, 0x3F, 0x90, 0xDF, 0x56, 0x6F, 0x23,
			0x47, 0x12, 0xAD, 0x32, 0x7A, 0xBC, 0x91, 0x36,
			0x43, 0xD2, 0x88, 0x3D, 0x2C, 0x31, 0x85, 0xE6,
			0x22, 0x2E, 0xCF, 0x53, 0x87, 0x0D, 0xAE, 0x72,
			0x31, 0x21, 0x52, 0x0C, 0xDF, 0xAC, 0xEA, 0x57
		},
		.len = 112
	},
	.modulus = {
		.data = {
			0x20, 0x7E, 0x78, 0xFC, 0x54, 0x40, 0x47, 0xED,
			0x7B, 0x26, 0x21, 0x94, 0x6D, 0x9C, 0xDC, 0xB7,
			0x7C, 0xB8, 0xDE, 0x57, 0x3C, 0x64, 0x47, 0x50,
			0xDB, 0x92, 0x0A, 0x5A, 0x85, 0x77, 0x84, 0xE3,
			0xC7, 0xEA, 0x33, 0xA2, 0x63, 0xDA, 0x63, 0xE3,
			0xDF, 0x07, 0x32, 0x85, 0xDC, 0xC3, 0xF0, 0x7C,
			0xD8, 0x44, 0xC4, 0xDE, 0x2C, 0xB7, 0x91, 0xAE,
			0xCD, 0xA2, 0xB3, 0x6B, 0x43, 0x04, 0x88, 0xBE,
			0x3B, 0x50, 0xE1, 0x88, 0xD0, 0x20, 0x04, 0x36,
			0xA4, 0xA0, 0xB0, 0x3B, 0x61, 0x9B, 0x83, 0xDB,
			0x05, 0x77, 0x5A, 0x5E, 0x87, 0xA6, 0xBE, 0x6A,
			0x2A, 0xB4, 0x30, 0x10, 0x8D, 0x3B, 0xBC, 0x84,
			0x9E, 0xB2, 0x21, 0x7E, 0xAC, 0x67, 0x94, 0x9F,
			0xF1, 0x32, 0x3C, 0xF0, 0x94, 0x83, 0xF8, 0x19
		},
		.len = 112
	},
	.result_len = 112
},
{
	.description = "Modular Exponentiation "
				   "(mod=299, base=240, exp=321, res=299)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0xAF, 0xE4, 0xCF, 0x9F, 0x7C, 0x20, 0x72, 0xE9,
			0x50, 0x2E, 0xE5, 0xE4, 0xEF, 0x80, 0x28, 0xB3,
			0x3A, 0x92, 0xAC, 0xCD, 0xF1, 0x6B, 0x7D, 0x83,
			0x78, 0x4E, 0x9B, 0x30, 0x1E, 0xF4, 0x11, 0x15,
			0x49, 0xAC, 0x06, 0xA2, 0x92, 0xB9, 0x91, 0x1B,
			0xE8, 0xC8, 0xBC, 0x8A, 0x6F, 0xB2, 0xB8, 0x7C,
			0xC5, 0xD3, 0x68, 0x60, 0xA4, 0x37, 0x7A, 0x6E,
			0x3A, 0x1C, 0xAE, 0xAC, 0x70, 0x7B, 0x03, 0xE0,
			0xDC, 0x7D, 0x51, 0x2E, 0x04, 0xC1, 0xD4, 0xA8,
			0x5A, 0xE1, 0xE8, 0xAD, 0x83, 0x0C, 0x0C, 0x2D,
			0x93, 0x8A, 0x07, 0x25, 0xC6, 0xE1, 0xAB, 0xD7,
			0x2F, 0xC4, 0x14, 0x2E, 0x68, 0x89, 0xA7, 0xEA,
			0x4E, 0x7B, 0xC4, 0x05, 0xD4, 0xB7, 0xC1, 0x9B,
			0x9D, 0x8D, 0x58, 0x33, 0xE3, 0xB0, 0x58, 0xD8,
			0xCF, 0x6D, 0xA9, 0xC6, 0x96, 0xE3, 0x6E, 0xAA,
			0x23, 0x17, 0x19, 0x74, 0xE0, 0x7B, 0x50, 0x7A,
			0x57, 0x49, 0xFD, 0xFA, 0x3E, 0x7D, 0xF9, 0xB6,
			0x30, 0x5F, 0x1C, 0xE4, 0x5F, 0xC7, 0x3D, 0x5B,
			0x9E, 0xF5, 0xB6, 0x2F, 0xEA, 0xCF, 0x31, 0x35,
			0xC0, 0x60, 0xDE, 0x18, 0xC5, 0x0D, 0xBB, 0xC5,
			0xD1, 0x1D, 0x25, 0x7C, 0x8C, 0x35, 0x8A, 0x71,
			0xA8, 0x01, 0x9E, 0xEA, 0x8F, 0xD4, 0x5D, 0x52,
			0x86, 0xB7, 0x11, 0xC8, 0xF2, 0x97, 0xD0, 0x73,
			0x7C, 0xAB, 0xBB, 0xF4, 0x38, 0x4E, 0x48, 0xB5,
			0x70, 0x35, 0x2A, 0xC5, 0x14, 0x86, 0x2E, 0x64,
			0x91, 0x32, 0x37, 0x5E, 0x1A, 0x00, 0xAC, 0xF1,
			0xFC, 0x36, 0xEA, 0x7F, 0x50, 0xAF, 0x02, 0xEC,
			0x06, 0xE8, 0x55, 0x68, 0x0D, 0x9A, 0x58, 0x4D,
			0xBD, 0xB9, 0x62, 0x25, 0xAB, 0x94, 0xD7, 0x37,
			0xAD, 0xB4, 0x9F, 0xB4, 0x3A, 0x07, 0x45, 0x4B
		},
		.len = 240
	},
	.exponent = {
		.data = {
			0xB1, 0xC2, 0x86, 0xFA, 0xE2, 0xF1, 0x71, 0x9C,
			0x61, 0x23, 0xAB, 0x37, 0xC6, 0x4F, 0x17, 0xFE,
			0x4D, 0xAC, 0x11, 0xD4, 0x36, 0xEE, 0xCB, 0xAE,
			0x46, 0x88, 0xA4, 0x92, 0x20, 0x6D, 0xDC, 0xF1,
			0xE4, 0x94, 0x72, 0x07, 0x64, 0x84, 0xF4, 0x83,
			0x31, 0x0C, 0x04, 0xF7, 0x5B, 0x68, 0xE6, 0x7A,
			0x6C, 0xCD, 0x6C, 0xBF, 0x03, 0x07, 0x5A, 0x91,
			0x37, 0x3A, 0x73, 0xFF, 0xB2, 0x11, 0x88, 0x39,
			0x19, 0xEB, 0x1C, 0x0E, 0x45, 0x99, 0xE6, 0x4E,
			0xE4, 0xB1, 0x57, 0xBE, 0xBE, 0x7A, 0xE8, 0x56,
			0x19, 0x92, 0xAC, 0xBD, 0x78, 0xCC, 0x54, 0xDC,
			0x2D, 0xE6, 0x7D, 0x61, 0xE1, 0x27, 0xA7, 0x43,
			0x46, 0x25, 0x51, 0x95, 0x47, 0xF6, 0xB1, 0x68,
			0x17, 0xE6, 0x21, 0xD4, 0x83, 0x1E, 0x32, 0xAF,
			0x22, 0xA4, 0x7D, 0x3D, 0x1F, 0xE6, 0x43, 0x96,
			0x64, 0xAB, 0xC5, 0x81, 0xBC, 0x79, 0x14, 0x54,
			0x02, 0x78, 0x79, 0x71, 0x58, 0xC2, 0x2E, 0x56,
			0x21, 0x6B, 0x40, 0xDB, 0x79, 0xD1, 0x80, 0x5D,
			0x61, 0xF0, 0x9F, 0x4A, 0xC3, 0x8F, 0xAC, 0x98,
			0x94, 0x88, 0x2C, 0xA5, 0xCB, 0x06, 0x47, 0x73,
			0x27, 0x71, 0xA8, 0x0C, 0xBD, 0xFD, 0x83, 0xBF,
			0xA2, 0xCC, 0x91, 0x63, 0x9D, 0xC5, 0x58, 0x50,
			0x53, 0x98, 0xA0, 0x5F, 0x0B, 0xDE, 0x15, 0x65,
			0xFB, 0x5D, 0xF1, 0x9C, 0xD0, 0xC3, 0x6B, 0x4D,
			0x31, 0x20, 0x2F, 0x4D, 0x4F, 0x9D, 0xEB, 0xCB,
			0xFC, 0xDA, 0x54, 0xC1, 0x57, 0x10, 0x0F, 0xFC,
			0xD2, 0xA7, 0x44, 0x0E, 0x89, 0x0D, 0x89, 0x56,
			0x1E, 0x40, 0x64, 0xFF, 0x9E, 0xB1, 0x5C, 0x9A,
			0x6E, 0xE5, 0xE9, 0x48, 0xAB, 0x27, 0x91, 0x9A,
			0x3B, 0x8D, 0xB6, 0xA0, 0xD6, 0xD8, 0x9B, 0xBD,
			0x0D, 0x1D, 0x90, 0xED, 0x54, 0xE1, 0x75, 0x5B,
			0x89, 0xE1, 0x0C, 0xC7, 0x42, 0xD7, 0x68, 0xCB,
			0x41, 0x59, 0xC1, 0x96, 0xD9, 0x77, 0x88, 0xF1,
			0x68, 0x90, 0xDA, 0xE0, 0xB7, 0x1E, 0x2C, 0xDB,
			0x27, 0x78, 0xC0, 0x15, 0x68, 0x9E, 0xF7, 0x48,
			0x45, 0xFC, 0x22, 0x5D, 0x2B, 0xFA, 0xC7, 0x81,
			0x26, 0x60, 0xF7, 0x50, 0xEE, 0xF9, 0x61, 0xF8,
			0x59, 0x28, 0x92, 0xEE, 0xD7, 0x92, 0x5F, 0x5E,
			0xA5, 0x5A, 0x4C, 0xC7, 0x89, 0x9B, 0x9F, 0x8F,
			0x01, 0x3B, 0x9D, 0x8D, 0xF8, 0x6B, 0xEE, 0x64,
			0x54
		},
		.len = 321
	},
	.reminder = {
		.data = {
			0x4A, 0x18, 0x9C, 0xCB, 0x90, 0x71, 0x8E, 0xD3,
			0xCA, 0xEB, 0xF1, 0xE7, 0xE8, 0xB0, 0x19, 0x9C,
			0x05, 0x51, 0x29, 0x8C, 0xB9, 0x6D, 0x1B, 0x05,
			0xDC, 0x16, 0x91, 0x0E, 0x69, 0xF3, 0x76, 0x29,
			0x9D, 0x91, 0x21, 0x98, 0x56, 0x9A, 0x22, 0x20,
			0xDF, 0x75, 0x4D, 0x40, 0x51, 0x99, 0x6E, 0xEA,
			0x37, 0x22, 0xF4, 0x27, 0x04, 0x6E, 0xDC, 0xB2,
			0xF5, 0xF6, 0xF8, 0xD6, 0xA8, 0xB7, 0x2D, 0xB7,
			0x18, 0x44, 0xF7, 0x62, 0x91, 0x44, 0x97, 0x91,
			0x6C, 0x14, 0x7E, 0xEB, 0x00, 0xB0, 0x3F, 0x7D,
			0x7B, 0x4A, 0xD0, 0x04, 0xD2, 0xCC, 0x5A, 0x22,
			0xB5, 0x0E, 0xAB, 0x1A, 0xB0, 0xD7, 0x97, 0xDD,
			0xE5, 0x78, 0xA9, 0x97, 0xF4, 0xC7, 0xE0, 0x28,
			0x00, 0xF3, 0x48, 0xCA, 0x69, 0xD0, 0xE8, 0x43,
			0x12, 0x82, 0x24, 0xBA, 0x28, 0xBD, 0x8E, 0xCB,
			0xB1, 0x98, 0x08, 0xC5, 0x0F, 0xF2, 0xE9, 0xA2,
			0x6C, 0xCC, 0xA9, 0x21, 0xA6, 0x38, 0xAE, 0x88,
			0x35, 0x5E, 0xBB, 0xEF, 0x37, 0xAB, 0xDA, 0x07,
			0x5F, 0x0A, 0xB4, 0x29, 0x65, 0x24, 0x22, 0x6C,
			0x9E, 0xF5, 0x19, 0xA4, 0x8E, 0x5A, 0xFA, 0xFC,
			0x97, 0x8A, 0xE8, 0x2B, 0x6D, 0x4B, 0xD0, 0xFB,
			0x86, 0xB9, 0xE7, 0x2C, 0x08, 0x25, 0x17, 0x90,
			0x77, 0x54, 0xE0, 0xBA, 0x0F, 0x59, 0x6C, 0x8C,
			0x0E, 0xCF, 0x54, 0x55, 0x6C, 0xF1, 0x65, 0x48,
			0xCC, 0xF4, 0xAB, 0xA1, 0x91, 0x07, 0x29, 0xFC,
			0x46, 0xBC, 0x2C, 0x85, 0xA1, 0x0C, 0x8A, 0x77,
			0x7A, 0xC6, 0x01, 0x34, 0xCE, 0x92, 0x1D, 0x88,
			0x54, 0x23, 0x26, 0x9B, 0x6B, 0x80, 0x6D, 0x08,
			0x99, 0xAE, 0xC0, 0xF6, 0x45, 0x97, 0xAF, 0xCD,
			0x2F, 0x4A, 0x7E, 0xAB, 0xD8, 0x31, 0x48, 0xA1,
			0xEB, 0x5E, 0xD1, 0xC0, 0xE7, 0xD5, 0x37, 0x3D,
			0x03, 0xA3, 0x16, 0x09, 0xD4, 0xDE, 0xC3, 0x97,
			0x13, 0xB6, 0x67, 0x55, 0x8A, 0x71, 0x51, 0x66,
			0xF5, 0xA1, 0x3B, 0xE3, 0x49, 0x8D, 0x7C, 0x52,
			0xCD, 0xA7, 0x11, 0xDD, 0xE0, 0xA0, 0x5C, 0xD8,
			0xF8, 0xDF, 0x01, 0xC5, 0x61, 0x87, 0xB4, 0xDE,
			0x3E, 0x39, 0xED, 0xC3, 0x3F, 0x84, 0x70, 0x37,
			0xBA, 0xDB, 0x5B
		},
		.len = 299
	},
	.modulus = {
		.data = {
			0x85, 0x04, 0x13, 0x7C, 0x4D, 0xBF, 0xC6, 0x25,
			0xD9, 0xAA, 0x1F, 0xED, 0x00, 0x69, 0xD7, 0x6C,
			0xB0, 0x46, 0x52, 0xA5, 0xF4, 0xF6, 0x55, 0x16,
			0x67, 0x52, 0x09, 0xF0, 0x28, 0xA7, 0x30, 0x22,
			0x34, 0xF1, 0xEA, 0xEB, 0x7C, 0x18, 0xEE, 0xAC,
			0x1A, 0xC1, 0xF5, 0x31, 0x7E, 0xA4, 0x4A, 0x0C,
			0xEA, 0xFE, 0x33, 0xDB, 0x49, 0x04, 0xFD, 0x33,
			0x3F, 0xB1, 0x41, 0x1F, 0xBD, 0x43, 0x71, 0xDE,
			0xB9, 0xA2, 0x4F, 0x20, 0x57, 0xAF, 0x27, 0x37,
			0x58, 0xA8, 0x51, 0x5D, 0x4E, 0xAB, 0x17, 0x1C,
			0x99, 0xD9, 0xB2, 0x0D, 0x21, 0xCA, 0x35, 0x52,
			0xF2, 0x4C, 0x7C, 0x79, 0x83, 0x2C, 0xF2, 0x87,
			0xC5, 0x58, 0x6E, 0x6E, 0x48, 0xFB, 0x32, 0x4D,
			0x1C, 0xDC, 0xE9, 0xDA, 0x9B, 0x77, 0x19, 0xD9,
			0x78, 0xE4, 0xF5, 0x3A, 0x49, 0x3D, 0x0D, 0x3D,
			0x10, 0x77, 0x0B, 0xC2, 0xE6, 0x66, 0x68, 0xFA,
			0x55, 0x99, 0x65, 0x5E, 0x55, 0x87, 0xCF, 0x3C,
			0x9C, 0x6C, 0x08, 0x09, 0x1F, 0x9C, 0xCB, 0x5E,
			0xE5, 0x19, 0x39, 0xA9, 0x2F, 0xF0, 0x49, 0x3D,
			0x7C, 0xB6, 0x7B, 0xA2, 0x93, 0xF1, 0x52, 0xD8,
			0x92, 0xDD, 0x56, 0x57, 0x8E, 0xE2, 0x5F, 0xA0,
			0x64, 0xB2, 0xC5, 0x0F, 0xB9, 0x89, 0xA6, 0x3F,
			0x54, 0x51, 0x2D, 0x01, 0x51, 0x78, 0x32, 0xE1,
			0xA7, 0x4D, 0x45, 0xC0, 0xD0, 0x6C, 0xE7, 0xCA,
			0xB2, 0x3F, 0x17, 0xD4, 0xB6, 0x58, 0x9B, 0xA8,
			0xBA, 0x2F, 0x3D, 0x1D, 0x6A, 0x73, 0x82, 0x2B,
			0x26, 0x2E, 0x7A, 0xEE, 0xEA, 0x41, 0x25, 0xFE,
			0xF0, 0xA2, 0x9C, 0x60, 0x35, 0xAD, 0x34, 0x30,
			0x55, 0x02, 0x6B, 0x06, 0xF4, 0xAD, 0x91, 0xA3,
			0xA2, 0x9C, 0x12, 0x8D, 0xDF, 0x2B, 0x3F, 0x0C,
			0x54, 0xCB, 0x98, 0xBA, 0xA1, 0x33, 0x70, 0xEF,
			0xF1, 0xEE, 0x15, 0xB7, 0xC6, 0x27, 0x47, 0x83,
			0x90, 0x58, 0x08, 0x16, 0x83, 0x94, 0xE9, 0x95,
			0x8B, 0x03, 0xD0, 0x3C, 0x45, 0xF8, 0x90, 0xC9,
			0xA7, 0x64, 0x76, 0xE8, 0x01, 0xA4, 0xA2, 0xAD,
			0x6F, 0x19, 0xCF, 0x38, 0x9A, 0xAB, 0x6E, 0xBE,
			0x79, 0xE4, 0x0F, 0xCE, 0x9C, 0x59, 0xF2, 0xF4,
			0x26, 0xAB, 0x0F
		},
		.len = 299
	},
	.result_len = 299
},
{
	.description = "Modular Exponentiation "
				   "(mod=448, base=50, exp=40, res=448)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0x92, 0x45, 0x17, 0x7D, 0xD3, 0xF4, 0x2B, 0x93,
			0x8E, 0x1A, 0xFB, 0x1D, 0x13, 0x55, 0x53, 0x84,
			0x96, 0x3C, 0x39, 0xE0, 0xAF, 0x4A, 0xB4, 0xC9,
			0x16, 0x1F, 0xF4, 0x24, 0x65, 0xDD, 0xC3, 0x62,
			0x12, 0xAF, 0x86, 0x95, 0x0D, 0xDE, 0x28, 0x87,
			0x90, 0x11, 0xAA, 0x6E, 0x60, 0xCD, 0x54, 0xB7,
			0x48, 0x43
		},
		.len = 50
	},
	.exponent = {
		.data = {
			0x22, 0xD9, 0x4D, 0x01, 0x2F, 0x50, 0x5D, 0xE1,
			0x01, 0xAA, 0xC6, 0xC6, 0xCD, 0x5D, 0x7E, 0x61,
			0x75, 0x0A, 0xDC, 0x06, 0x07, 0x4B, 0xBD, 0x29,
			0x33, 0x09, 0x91, 0xD4, 0x29, 0xEB, 0x52, 0x24,
			0x27, 0xC6, 0x83, 0x6D, 0x70, 0xA9, 0xC9, 0x11
		},
		.len = 40
	},
	.reminder = {
		.data = {
			0x6B, 0x15, 0x66, 0xD3, 0x81, 0x3E, 0xCB, 0xA6,
			0xD8, 0x88, 0x9C, 0xFF, 0x8D, 0xBC, 0x91, 0x9F,
			0xDA, 0x09, 0x9E, 0x1B, 0xDB, 0x8A, 0x0F, 0xAC,
			0x19, 0x8E, 0xFE, 0x25, 0xA0, 0x66, 0x9D, 0x19,
			0x7C, 0xCB, 0x1D, 0xCF, 0x97, 0x41, 0xD2, 0x7A,
			0x0E, 0xAE, 0x8A, 0xD4, 0x24, 0x6D, 0xDF, 0xF4,
			0x0A, 0xCD, 0xE7, 0xF3, 0x21, 0x76, 0x22, 0xCA,
			0x31, 0x67, 0x91, 0x1B, 0x3A, 0x38, 0xAF, 0x03,
			0xA6, 0xFF, 0x34, 0xA2, 0x92, 0xC8, 0x70, 0xD4,
			0x86, 0xA5, 0x32, 0x57, 0xDC, 0xE5, 0xEB, 0xA5,
			0x33, 0xC2, 0x83, 0xC3, 0x33, 0xE8, 0x53, 0x8F,
			0x8A, 0xFA, 0xAE, 0xCF, 0x35, 0x6E, 0xD0, 0x5D,
			0xCF, 0xE4, 0x18, 0x2F, 0x6E, 0xEC, 0x9D, 0xE9,
			0x60, 0x27, 0xF7, 0x7A, 0xFD, 0x1D, 0xC3, 0x2C,
			0xE0, 0xDA, 0x7B, 0x18, 0xE0, 0x6A, 0x06, 0xE6,
			0xEE, 0xA3, 0x0C, 0x02, 0x98, 0xA5, 0x30, 0xA1,
			0x3F, 0x98, 0xCA, 0x06, 0xBD, 0x76, 0xDB, 0xCC,
			0xDF, 0x0B, 0x38, 0x43, 0x86, 0x11, 0xB8, 0xB7,
			0xC5, 0xC1, 0x74, 0x59, 0x63, 0xFF, 0x3C, 0xFC,
			0x14, 0xDF, 0xAD, 0xDD, 0x99, 0xCC, 0x48, 0xEA,
			0xCE, 0x75, 0x15, 0x1D, 0x8F, 0x63, 0x5B, 0x9D,
			0x97, 0x2E, 0x98, 0x7B, 0x7A, 0x07, 0xBB, 0x99,
			0x57, 0xF5, 0xC4, 0x4D, 0x11, 0x4C, 0x55, 0x08,
			0x5F, 0x2C, 0x25, 0x37, 0x83, 0x76, 0xD5, 0xCA,
			0x51, 0x5C, 0x86, 0xA4, 0xA1, 0x14, 0x2B, 0xBA,
			0xBD, 0x59, 0xBF, 0xE0, 0x41, 0xA9, 0x66, 0xD1,
			0x29, 0x4B, 0x04, 0xCA, 0xEE, 0x9A, 0x31, 0x5E,
			0xD9, 0x11, 0xEE, 0x1B, 0xB6, 0x09, 0x11, 0xA4,
			0xD8, 0x3A, 0x12, 0xF5, 0x9E, 0x79, 0x02, 0x64,
			0x8F, 0x18, 0x38, 0x5D, 0x19, 0x4D, 0x56, 0xA6,
			0x7A, 0xA6, 0xC0, 0x98, 0x13, 0x2F, 0x60, 0x73,
			0xD2, 0x8F, 0x70, 0x3D, 0x09, 0x42, 0xE9, 0xCC,
			0xFF, 0xA9, 0xF0, 0x82, 0xA2, 0x10, 0x1A, 0x26,
			0x91, 0xD6, 0x84, 0x24, 0x0A, 0x3A, 0xC6, 0x67,
			0x36, 0xCE, 0x9B, 0x4B, 0xF8, 0xF3, 0xCA, 0xB6,
			0x6C, 0x5F, 0x04, 0xE5, 0x74, 0xD8, 0x56, 0x42,
			0x7D, 0xD1, 0x5E, 0x8F, 0xA0, 0x9D, 0x66, 0x3B,
			0x13, 0x86, 0x9D, 0x84, 0x40, 0x74, 0x76, 0x00,
			0xCD, 0x40, 0x87, 0x58, 0x7C, 0x41, 0x8A, 0xE2,
			0xF8, 0xCB, 0x06, 0xCF, 0xCB, 0x0C, 0x78, 0x5D,
			0x6E, 0xAD, 0xE0, 0x1A, 0x54, 0xC3, 0x31, 0x5B,
			0xCF, 0xD0, 0x2D, 0x22, 0xA5, 0x84, 0x97, 0x20,
			0x4D, 0x13, 0x5F, 0xED, 0xFE, 0x01, 0xD8, 0x1A,
			0x79, 0xB5, 0x99, 0x03, 0x18, 0x55, 0xFB, 0xDD,
			0x5F, 0x6E, 0x85, 0xA2, 0x13, 0x28, 0x45, 0x29,
			0x25, 0x71, 0xC8, 0x53, 0x9B, 0x45, 0xA5, 0xD9,
			0x55, 0x8C, 0xC8, 0x93, 0x2B, 0xBC, 0x08, 0x8D,
			0x47, 0x90, 0x8F, 0x5E, 0x4B, 0xBC, 0xCE, 0xD1,
			0x51, 0x3B, 0xFB, 0x09, 0x4A, 0x72, 0x99, 0x0F,
			0xE2, 0x5F, 0x80, 0xF4, 0xE7, 0x6F, 0xD6, 0x08,
			0xD1, 0x79, 0xE8, 0x68, 0xFA, 0x29, 0x08, 0x4F,
			0x8B, 0x11, 0x12, 0x27, 0x97, 0x04, 0xC8, 0x7D,
			0xAB, 0x4D, 0xDA, 0x92, 0x29, 0x42, 0x7C, 0x3E,
			0x7D, 0x05, 0xEC, 0x1F, 0x4A, 0xA1, 0xBB, 0x3F,
			0x77, 0x3F, 0xEF, 0xC9, 0xAA, 0x63, 0xD5, 0x39,
			0xD5, 0x33, 0x0D, 0x28, 0xD0, 0x3C, 0x85, 0xC0
		},
		.len = 448
	},
	.modulus = {
		.data = {
			0xBB, 0x97, 0x8A, 0xB6, 0x26, 0xD4, 0x0E, 0x70,
			0x21, 0xA6, 0x56, 0x71, 0xE5, 0xD8, 0x18, 0x21,
			0x64, 0x9F, 0x1B, 0x6F, 0x7C, 0x27, 0x72, 0xB8,
			0x39, 0xE5, 0x2A, 0x94, 0x76, 0x22, 0xB7, 0x68,
			0x57, 0x3A, 0x01, 0x54, 0xA8, 0x50, 0x41, 0xA1,
			0xAD, 0xD0, 0xC7, 0xDB, 0xAA, 0x76, 0x7F, 0x37,
			0xA9, 0x27, 0x22, 0x8D, 0xF0, 0x5C, 0x5A, 0xAC,
			0xFB, 0x82, 0x6A, 0x8E, 0x31, 0x51, 0x54, 0x7C,
			0xDB, 0x55, 0x9C, 0xBC, 0x82, 0x27, 0xF4, 0x0B,
			0x94, 0x74, 0xC8, 0x83, 0x80, 0x1C, 0xD8, 0xFF,
			0x50, 0xA2, 0xC9, 0xED, 0x2B, 0x98, 0x77, 0xF3,
			0x31, 0x81, 0x1C, 0x41, 0x8E, 0xAF, 0x87, 0xA2,
			0x02, 0xAC, 0x8B, 0x55, 0x01, 0x5C, 0x16, 0x11,
			0x63, 0x8C, 0xE2, 0x0D, 0x51, 0xD2, 0x4C, 0xD7,
			0xD4, 0x3D, 0xE4, 0x79, 0x1A, 0xA7, 0xC4, 0xBF,
			0x4E, 0x2A, 0xC9, 0x74, 0xD6, 0xD4, 0x90, 0x03,
			0x65, 0x7F, 0x54, 0x0F, 0xAC, 0x5F, 0x98, 0x2C,
			0x46, 0xC0, 0xD7, 0xE6, 0x75, 0x95, 0xC3, 0xEA,
			0x05, 0x3A, 0x03, 0x55, 0x43, 0xC7, 0xC2, 0xD1,
			0x11, 0xCD, 0x57, 0x37, 0x0D, 0x40, 0x87, 0xDF,
			0x7D, 0xC3, 0x04, 0x54, 0xDE, 0x1D, 0xAF, 0xB8,
			0x02, 0x50, 0x42, 0xFF, 0x9D, 0xFB, 0x13, 0xF7,
			0x25, 0x5A, 0x8B, 0xE3, 0x31, 0xA2, 0x64, 0xF8,
			0x94, 0x50, 0x18, 0xFB, 0xBB, 0xA2, 0xE9, 0x13,
			0x77, 0x6E, 0xE1, 0x6F, 0x9F, 0x06, 0x03, 0xEE,
			0x0D, 0x06, 0x6E, 0xF2, 0x9B, 0x15, 0x70, 0xDD,
			0x26, 0x7C, 0xB4, 0x5D, 0xD0, 0xE7, 0x77, 0xC4,
			0xB9, 0x88, 0x75, 0xB8, 0x73, 0xFC, 0xE9, 0xB7,
			0x11, 0x26, 0xAC, 0xDB, 0x97, 0x27, 0x18, 0x21,
			0x50, 0x7E, 0x46, 0xB2, 0xF1, 0x50, 0x40, 0xD9,
			0x8B, 0x63, 0xDB, 0x1A, 0x8E, 0x29, 0xBE, 0x1F,
			0x88, 0x35, 0xFD, 0x95, 0xC3, 0xA6, 0x80, 0xEB,
			0x73, 0xF7, 0x02, 0x02, 0xB0, 0xCA, 0x97, 0x2C,
			0x32, 0x44, 0xA9, 0xCA, 0x94, 0xC0, 0xB2, 0xD9,
			0x7C, 0xD9, 0x10, 0x62, 0x31, 0xC9, 0xFA, 0x5B,
			0x5C, 0x2C, 0xB6, 0x04, 0x5B, 0x7E, 0x86, 0xBB,
			0x49, 0x02, 0x16, 0x9E, 0x1E, 0x53, 0xBD, 0xC2,
			0xA9, 0xAA, 0x94, 0x39, 0xA1, 0xB2, 0x18, 0x17,
			0xB6, 0x2C, 0xF6, 0xFF, 0xC0, 0xD0, 0x2D, 0x4D,
			0xAA, 0x6C, 0xB5, 0xC8, 0x6A, 0xBE, 0x38, 0xE4,
			0x9F, 0xDC, 0x5C, 0x56, 0x56, 0x04, 0x32, 0x49,
			0x91, 0x17, 0x44, 0x6E, 0xB3, 0xF9, 0x8F, 0xDB,
			0xEA, 0x04, 0x4C, 0x67, 0xE6, 0xDA, 0x96, 0x2F,
			0x89, 0x2D, 0x54, 0xC3, 0xAD, 0x07, 0x6B, 0xA0,
			0x87, 0xF0, 0xCF, 0x4B, 0x43, 0x46, 0xCA, 0x06,
			0x48, 0x8D, 0x34, 0xC4, 0xD2, 0xD2, 0xA9, 0x16,
			0x13, 0xF5, 0x49, 0x44, 0x8C, 0xD9, 0x0C, 0x1E,
			0x79, 0x47, 0xFC, 0x4C, 0x37, 0x8E, 0xD1, 0xFD,
			0xB9, 0xE8, 0x6E, 0x1B, 0x8D, 0x68, 0xCC, 0x49,
			0x0D, 0x98, 0xB5, 0xD5, 0x22, 0x1C, 0xFC, 0xBA,
			0x7A, 0x74, 0x3D, 0xBD, 0xD9, 0xB3, 0x80, 0x58,
			0x4A, 0x05, 0x67, 0x9D, 0x59, 0xF4, 0xF7, 0x72,
			0x11, 0x3C, 0x67, 0x96, 0xE7, 0x0D, 0x8E, 0x73,
			0xD1, 0xEE, 0x00, 0x79, 0x98, 0x7A, 0x0E, 0xE0,
			0xA8, 0xEA, 0x7D, 0xF3, 0xDB, 0x0E, 0x62, 0x3E,
			0x66, 0x95, 0xED, 0xD5, 0x8C, 0x39, 0xF5, 0xAB
		},
		.len = 448
	},
	.result_len = 448
},
{
	.description = "Modular Exponentiation "
				   "(mod=19, base=500, exp=35, res=19)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.base = {
		.data = {
			0x42, 0xE5, 0x6F, 0xF7, 0xEF, 0x8F, 0xAB, 0x6F,
			0xF5, 0xE2, 0xD2, 0x97, 0x68, 0x0D, 0x52, 0xDD,
			0x3D, 0x62, 0xC4, 0xC4, 0xDE, 0xD2, 0x07, 0x50,
			0x1B, 0xA9, 0x5F, 0xAE, 0x42, 0xCB, 0x5E, 0x0B,
			0xC0, 0x41, 0xFE, 0xEF, 0x22, 0xA6, 0x4E, 0x35,
			0x80, 0x61, 0x22, 0x92, 0x65, 0x77, 0x45, 0x84,
			0x6C, 0x03, 0x7C, 0xEF, 0xC0, 0x49, 0x31, 0x87,
			0x86, 0x9B, 0x2E, 0x28, 0xA1, 0x55, 0x2D, 0x3C,
			0x68, 0x3D, 0x69, 0x0E, 0x98, 0xD6, 0x40, 0xFD,
			0x0B, 0x4C, 0x5F, 0xC7, 0x95, 0xF1, 0x53, 0x7C,
			0xCC, 0x12, 0x3F, 0x8C, 0x7A, 0x24, 0x73, 0xE7,
			0x33, 0x20, 0xBD, 0x0C, 0xD2, 0x9C, 0x12, 0x20,
			0xC0, 0xC0, 0xA9, 0x16, 0x7E, 0x6B, 0x1D, 0x1F,
			0xFE, 0x63, 0x8A, 0x22, 0x75, 0xDC, 0xF0, 0x0F,
			0x8E, 0xA4, 0x3C, 0xE3, 0x6B, 0xFA, 0x46, 0xBA,
			0xEB, 0x91, 0x31, 0x9D, 0x34, 0xED, 0xA0, 0xA6,
			0xA3, 0xAB, 0xA5, 0x2A, 0x10, 0x30, 0xF4, 0x6F,
			0x80, 0x25, 0xB0, 0xF5, 0x56, 0x76, 0xD3, 0xC5,
			0x10, 0x92, 0xCD, 0xEA, 0xC3, 0x9C, 0x52, 0x96,
			0xF1, 0xBD, 0x42, 0x53, 0xF3, 0xA9, 0x1F, 0xCB,
			0x53, 0x45, 0xF1, 0xF6, 0x5F, 0x98, 0xFC, 0x13,
			0xC9, 0xA8, 0x44, 0xC6, 0xD0, 0x78, 0xB6, 0x39,
			0x93, 0x02, 0xC6, 0xC9, 0x0F, 0xAF, 0xF9, 0x6D,
			0x91, 0x35, 0xC9, 0x26, 0x73, 0x11, 0xEB, 0xEE,
			0x52, 0x61, 0x6C, 0xC0, 0x7F, 0xFD, 0xD0, 0x77,
			0x9F, 0xC6, 0x0A, 0x05, 0x1B, 0x90, 0x61, 0x54,
			0x61, 0xFF, 0x1E, 0xBA, 0x1D, 0x2F, 0x25, 0xE9,
			0x85, 0x4C, 0xBA, 0xEF, 0x99, 0x95, 0x3C, 0xBB,
			0xA9, 0xCF, 0x0D, 0xBF, 0x2C, 0x86, 0xB0, 0x59,
			0xAA, 0x83, 0x29, 0x32, 0x24, 0x28, 0xC8, 0x53,
			0x28, 0x4C, 0xEB, 0x08, 0xFF, 0xC6, 0x25, 0xB7,
			0xFF, 0x18, 0xB0, 0x2C, 0xDD, 0xAE, 0xFB, 0xDB,
			0x54, 0xA2, 0x92, 0x27, 0x15, 0x0D, 0x6B, 0x50,
			0xFB, 0xEA, 0x2C, 0x1C, 0x6F, 0x91, 0x3C, 0x50,
			0x5A, 0xD8, 0x9B, 0x33, 0xED, 0x51, 0x5D, 0x7C,
			0x37, 0x01, 0xEF, 0x09, 0xEA, 0x59, 0x56, 0x8A,
			0x67, 0x21, 0x8C, 0x25, 0x00, 0x33, 0x24, 0x31,
			0xCB, 0xAA, 0x5A, 0xA4, 0xB1, 0x84, 0xDD, 0x89,
			0x3A, 0xFA, 0xD8, 0xAB, 0xEE, 0x3E, 0xC0, 0x0D,
			0xDA, 0x2C, 0x2A, 0x75, 0x13, 0xD7, 0x49, 0x5E,
			0x28, 0x2C, 0x24, 0xC6, 0x1E, 0xA0, 0xB3, 0x70,
			0xAD, 0x45, 0x8A, 0xF2, 0xD9, 0x38, 0x69, 0xD5,
			0x53, 0x30, 0xD8, 0x09, 0x09, 0xDE, 0x1F, 0x6C,
			0x36, 0x82, 0xD5, 0xEC, 0xA7, 0x7E, 0x37, 0x5F,
			0x7D, 0xF2, 0x85, 0x85, 0xF9, 0x0A, 0xC1, 0x13,
			0x86, 0x3C, 0xCD, 0xFE, 0x44, 0x46, 0x57, 0x5E,
			0x67, 0x39, 0x9D, 0x65, 0x74, 0xB9, 0x13, 0x5A,
			0x05, 0xC1, 0xEA, 0xB5, 0x10, 0x1D, 0x66, 0xFF,
			0xA0, 0x3D, 0x47, 0x27, 0x15, 0x66, 0x52, 0x19,
			0xFA, 0x95, 0xD0, 0x03, 0x67, 0xA8, 0x89, 0xAA,
			0x68, 0x04, 0x5F, 0xC4, 0x57, 0x1F, 0x6C, 0xF7,
			0xD2, 0xE2, 0xA3, 0xF3, 0x96, 0x70, 0x86, 0xC6,
			0xCB, 0x3D, 0x52, 0x66, 0x79, 0xED, 0xE6, 0x35,
			0x0A, 0xE8, 0xA8, 0x5A, 0xED, 0x41, 0xB0, 0xF0,
			0x89, 0xCC, 0x20, 0xDA, 0xB7, 0x48, 0x44, 0x64,
			0x69, 0xC9, 0x43, 0xE2, 0xBD, 0xD1, 0x17, 0xCF,
			0x25, 0x7C, 0x92, 0x0B, 0xFC, 0x71, 0x46, 0x67,
			0x1F, 0xF4, 0xA7, 0xFF, 0xD8, 0xA4, 0x5F, 0x4A,
			0x8A, 0x45, 0xBE, 0xDD, 0x89, 0xE2, 0x2A, 0xA7,
			0xBC, 0xE9, 0x84, 0x53, 0x9D, 0xF4, 0x39, 0xB7,
			0xDB, 0x3A, 0x17, 0xF7, 0x27, 0x39, 0xDF, 0x8A,
			0xF0, 0x72, 0xD6, 0x23, 0x1B, 0x1C, 0xD9, 0x48,
			0xF4, 0x2E, 0x54, 0xA3
		},
		.len = 500
	},
	.exponent = {
		.data = {
			0xE7, 0xF1, 0x97, 0x29, 0x62, 0x0B, 0x99, 0x89,
			0x99, 0xC1, 0x63, 0xA2, 0xB7, 0x29, 0xAD, 0x0E,
			0x84, 0x3B, 0x86, 0x82, 0xC4, 0xDD, 0xC4, 0xE2,
			0xA7, 0xD4, 0xBA, 0x91, 0x2C, 0xB5, 0xD6, 0xD4,
			0x74, 0x1D, 0xE1
		},
		.len = 35
	},
	.reminder = {
		.data = {
			0x67, 0x9F, 0xF7, 0x57, 0xD7, 0xF8, 0xF8, 0x90,
			0x4E, 0xB5, 0x34, 0xE8, 0xAF, 0x14, 0xC6, 0x94,
			0x5F, 0xA1, 0x03
		},
		.len = 19
	},
	.modulus = {
		.data = {
			0x9C, 0xE7, 0xE7, 0x14, 0x6E, 0x07, 0x71, 0xD2,
			0xD1, 0xB3, 0x59, 0x9B, 0x63, 0xDB, 0x58, 0x8D,
			0x5E, 0x84, 0xA0
		},
		.len = 19
	},
	.result_len = 19
}
};

static const struct
modinv_test_data modinv_test_case[] = {
{
	.description = "Modular Inverse (mod=128, base=20, exp=3, inv=128)",
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODINV,
	.base = {
		.data = {
			0xF8, 0xBA, 0x1A, 0x55, 0xD0, 0x2F, 0x85,
			0xAE, 0x96, 0x7B, 0xB6, 0x2F, 0xB6, 0xCD,
			0xA8, 0xEB, 0x7E, 0x78, 0xA0, 0x50
		},
		.len = 20
	},
	.inverse = {
		.data = {
			0x52, 0xb1, 0xa3, 0x8c, 0xc5, 0x8a, 0xb9, 0x1f,
			0xb6, 0x82, 0xf5, 0x6a, 0x9a, 0xde, 0x8d, 0x2e,
			0x62, 0x4b, 0xac, 0x49, 0x21, 0x1d, 0x30, 0x4d,
			0x32, 0xac, 0x1f, 0x40, 0x6d, 0x52, 0xc7, 0x9b,
			0x6c, 0x0a, 0x82, 0x3a, 0x2c, 0xaf, 0x6b, 0x6d,
			0x17, 0xbe, 0x43, 0xed, 0x97, 0x78, 0xeb, 0x4c,
			0x92, 0x6f, 0xcf, 0xed, 0xb1, 0x09, 0xcb, 0x27,
			0xc2, 0xde, 0x62, 0xfd, 0x21, 0xe6, 0xbd, 0x4f,
			0xfe, 0x7a, 0x1b, 0x50, 0xfe, 0x10, 0x4a, 0xb0,
			0xb7, 0xcf, 0xdb, 0x7d, 0xca, 0xc2, 0xf0, 0x1c,
			0x39, 0x48, 0x6a, 0xb5, 0x4d, 0x8c, 0xfe, 0x63,
			0x91, 0x9c, 0x21, 0xc3, 0x0e, 0x76, 0xad, 0x44,
			0x8d, 0x54, 0x33, 0x99, 0xe1, 0x80, 0x19, 0xba,
			0xb5, 0xac, 0x7d, 0x9c, 0xce, 0x91, 0x2a, 0xd9,
			0x2c, 0xe1, 0x16, 0xd6, 0xd7, 0xcf, 0x9d, 0x05,
			0x9a, 0x66, 0x9a, 0x3a, 0xc1, 0xb8, 0x4b, 0xc3
		},
		.len = 128
	},
	.modulus = {
		.data = {
			0xb3, 0xa1, 0xaf, 0xb7, 0x13, 0x08, 0x00, 0x0a,
			0x35, 0xdc, 0x2b, 0x20, 0x8d, 0xa1, 0xb5, 0xce,
			0x47, 0x8a, 0xc3, 0x80, 0xf4, 0x7d, 0x4a, 0xa2,
			0x62, 0xfd, 0x61, 0x7f, 0xb5, 0xa8, 0xde, 0x0a,
			0x17, 0x97, 0xa0, 0xbf, 0xdf, 0x56, 0x5a, 0x3d,
			0x51, 0x56, 0x4f, 0x70, 0x70, 0x3f, 0x63, 0x6a,
			0x44, 0x5b, 0xad, 0x84, 0x0d, 0x3f, 0x27, 0x6e,
			0x3b, 0x34, 0x91, 0x60, 0x14, 0xb9, 0xaa, 0x72,
			0xfd, 0xa3, 0x64, 0xd2, 0x03, 0xa7, 0x53, 0x87,
			0x9e, 0x88, 0x0b, 0xc1, 0x14, 0x93, 0x1a, 0x62,
			0xff, 0xb1, 0x5d, 0x74, 0xcd, 0x59, 0x63, 0x18,
			0x11, 0x3d, 0x4f, 0xba, 0x75, 0xd4, 0x33, 0x4e,
			0x23, 0x6b, 0x7b, 0x57, 0x44, 0xe1, 0xd3, 0x03,
			0x13, 0xa6, 0xf0, 0x8b, 0x60, 0xb0, 0x9e, 0xee,
			0x75, 0x08, 0x9d, 0x71, 0x63, 0x13, 0xcb, 0xa6,
			0x81, 0x92, 0x14, 0x03, 0x22, 0x2d, 0xde, 0x55
		},
		.len = 128
	},
	.result_len = 128
}
};

/* modular operation test data */
uint8_t base[] = {
	0xF8, 0xBA, 0x1A, 0x55, 0xD0, 0x2F, 0x85,
	0xAE, 0x96, 0x7B, 0xB6, 0x2F, 0xB6, 0xCD,
	0xA8, 0xEB, 0x7E, 0x78, 0xA0, 0x50
};

/* MODEX data. 8< */
uint8_t mod_p[] = {
	0x00, 0xb3, 0xa1, 0xaf, 0xb7, 0x13, 0x08, 0x00,
	0x0a, 0x35, 0xdc, 0x2b, 0x20, 0x8d, 0xa1, 0xb5,
	0xce, 0x47, 0x8a, 0xc3, 0x80, 0xf4, 0x7d, 0x4a,
	0xa2, 0x62, 0xfd, 0x61, 0x7f, 0xb5, 0xa8, 0xde,
	0x0a, 0x17, 0x97, 0xa0, 0xbf, 0xdf, 0x56, 0x5a,
	0x3d, 0x51, 0x56, 0x4f, 0x70, 0x70, 0x3f, 0x63,
	0x6a, 0x44, 0x5b, 0xad, 0x84, 0x0d, 0x3f, 0x27,
	0x6e, 0x3b, 0x34, 0x91, 0x60, 0x14, 0xb9, 0xaa,
	0x72, 0xfd, 0xa3, 0x64, 0xd2, 0x03, 0xa7, 0x53,
	0x87, 0x9e, 0x88, 0x0b, 0xc1, 0x14, 0x93, 0x1a,
	0x62, 0xff, 0xb1, 0x5d, 0x74, 0xcd, 0x59, 0x63,
	0x18, 0x11, 0x3d, 0x4f, 0xba, 0x75, 0xd4, 0x33,
	0x4e, 0x23, 0x6b, 0x7b, 0x57, 0x44, 0xe1, 0xd3,
	0x03, 0x13, 0xa6, 0xf0, 0x8b, 0x60, 0xb0, 0x9e,
	0xee, 0x75, 0x08, 0x9d, 0x71, 0x63, 0x13, 0xcb,
	0xa6, 0x81, 0x92, 0x14, 0x03, 0x22, 0x2d, 0xde,
	0x55
};

uint8_t mod_e[] = {0x01, 0x00, 0x01};
/* >8 End of MODEX data. */

/* Precomputed modular exponentiation for verification */
uint8_t mod_exp[] = {
	0x2C, 0x60, 0x75, 0x45, 0x98, 0x9D, 0xE0, 0x72,
	0xA0, 0x9D, 0x3A, 0x9E, 0x03, 0x38, 0x73, 0x3C,
	0x31, 0x83, 0x04, 0xFE, 0x75, 0x43, 0xE6, 0x17,
	0x5C, 0x01, 0x29, 0x51, 0x69, 0x33, 0x62, 0x2D,
	0x78, 0xBE, 0xAE, 0xC4, 0xBC, 0xDE, 0x7E, 0x2C,
	0x77, 0x84, 0xF2, 0xC5, 0x14, 0xB5, 0x2F, 0xF7,
	0xC5, 0x94, 0xEF, 0x86, 0x75, 0x75, 0xB5, 0x11,
	0xE5, 0x0E, 0x0A, 0x29, 0x76, 0xE2, 0xEA, 0x32,
	0x0E, 0x43, 0x77, 0x7E, 0x2C, 0x27, 0xAC, 0x3B,
	0x86, 0xA5, 0xDB, 0xC9, 0x48, 0x40, 0xE8, 0x99,
	0x9A, 0x0A, 0x3D, 0xD6, 0x74, 0xFA, 0x2E, 0x2E,
	0x5B, 0xAF, 0x8C, 0x99, 0x44, 0x2A, 0x67, 0x38,
	0x27, 0x41, 0x59, 0x9D, 0xB8, 0x51, 0xC9, 0xF7,
	0x43, 0x61, 0x31, 0x6E, 0xF1, 0x25, 0x38, 0x7F,
	0xAE, 0xC6, 0xD0, 0xBB, 0x29, 0x76, 0x3F, 0x46,
	0x2E, 0x1B, 0xE4, 0x67, 0x71, 0xE3, 0x87, 0x5A
};

/* Precomputed modular inverse for verification */
uint8_t mod_inv[] = {
	0x52, 0xb1, 0xa3, 0x8c, 0xc5, 0x8a, 0xb9, 0x1f,
	0xb6, 0x82, 0xf5, 0x6a, 0x9a, 0xde, 0x8d, 0x2e,
	0x62, 0x4b, 0xac, 0x49, 0x21, 0x1d, 0x30, 0x4d,
	0x32, 0xac, 0x1f, 0x40, 0x6d, 0x52, 0xc7, 0x9b,
	0x6c, 0x0a, 0x82, 0x3a, 0x2c, 0xaf, 0x6b, 0x6d,
	0x17, 0xbe, 0x43, 0xed, 0x97, 0x78, 0xeb, 0x4c,
	0x92, 0x6f, 0xcf, 0xed, 0xb1, 0x09, 0xcb, 0x27,
	0xc2, 0xde, 0x62, 0xfd, 0x21, 0xe6, 0xbd, 0x4f,
	0xfe, 0x7a, 0x1b, 0x50, 0xfe, 0x10, 0x4a, 0xb0,
	0xb7, 0xcf, 0xdb, 0x7d, 0xca, 0xc2, 0xf0, 0x1c,
	0x39, 0x48, 0x6a, 0xb5, 0x4d, 0x8c, 0xfe, 0x63,
	0x91, 0x9c, 0x21, 0xc3, 0x0e, 0x76, 0xad, 0x44,
	0x8d, 0x54, 0x33, 0x99, 0xe1, 0x80, 0x19, 0xba,
	0xb5, 0xac, 0x7d, 0x9c, 0xce, 0x91, 0x2a, 0xd9,
	0x2c, 0xe1, 0x16, 0xd6, 0xd7, 0xcf, 0x9d, 0x05,
	0x9a, 0x66, 0x9a, 0x3a, 0xc1, 0xb8, 0x4b, 0xc3
};

/* MODEX vector. 8< */
struct rte_crypto_asym_xform modex_xform = {
	.next = NULL,
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODEX,
	.modex = {
		.modulus = {
			.data = mod_p,
			.length = sizeof(mod_p)
		},
		.exponent = {
			.data = mod_e,
			.length = sizeof(mod_e)
		}
	}
};
/* >8 End of MODEX vector. */

struct rte_crypto_asym_xform modinv_xform = {
	.next = NULL,
	.xform_type = RTE_CRYPTO_ASYM_XFORM_MODINV,
	.modinv = {
		.modulus = {
			.data = mod_p,
			.length = sizeof(mod_p)
		}
	}
};

#endif /* TEST_CRYPTODEV_MOD_TEST_VECTORS_H__ */
