/*
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef STATE
struct {
	uint8_t akku;
	uint8_t x;
	uint8_t y;
	uint8_t sp;
	uint8_t flag_n;
	uint8_t flag_v;
	uint8_t flag_b;
	uint8_t flag_d;
	uint8_t flag_i;
	uint8_t flag_z;
	uint8_t flag_c;
	uint16_t pc;
	uint16_t addr;
	uint8_t t0;
	uint8_t t1;
	uint8_t t2;
	jmp_buf jmp_env;
} NAME;
#endif /* STATE */

#ifdef BEHAVIOR

static uint8_t
NAME_(mfb)(struct cpssp *cpssp)
{
	uint8_t byte = 0;

	byte = CHIP_(mrb)(cpssp, cpssp->NAME.pc);
	cpssp->NAME.pc++;
	return byte;
}

static void
NAME_(addr_l)(struct cpssp *cpssp)
{
	uint8_t l;

	l = NAME_(mfb)(cpssp);
	cpssp->NAME.addr = (uint16_t) l;
}

static void
NAME_(addr_hl)(struct cpssp *cpssp)
{
	uint8_t l;
	uint8_t h;

	l = NAME_(mfb)(cpssp);
	h = NAME_(mfb)(cpssp);
	cpssp->NAME.addr = ((uint16_t) h << 8) | (uint16_t) l;
}

static void
NAME_(addr_l_x)(struct cpssp *cpssp)
{
	NAME_(addr_l)(cpssp);
	cpssp->NAME.addr += cpssp->NAME.x;
}

static void
NAME_(addr_hl_x)(struct cpssp *cpssp)
{
	NAME_(addr_hl)(cpssp);
	cpssp->NAME.addr += cpssp->NAME.x;
}

static void
NAME_(addr_l_y)(struct cpssp *cpssp)
{
	NAME_(addr_l)(cpssp);
	cpssp->NAME.addr += cpssp->NAME.y;
}

static void
NAME_(addr_hl_y)(struct cpssp *cpssp)
{
	NAME_(addr_hl)(cpssp);
	cpssp->NAME.addr += cpssp->NAME.y;
}

static void
NAME_(addr_ind_l_x)(struct cpssp *cpssp)
{
	uint8_t l;
	uint8_t h;

	NAME_(addr_l)(cpssp);
	cpssp->NAME.addr += cpssp->NAME.x;
	l = CHIP_(mrb)(cpssp, cpssp->NAME.addr + 0);
	h = CHIP_(mrb)(cpssp, cpssp->NAME.addr + 1);
	cpssp->NAME.addr = ((uint16_t) h << 8) | (uint16_t) l;
}

static void
NAME_(addr_ind_l_y)(struct cpssp *cpssp)
{
	uint8_t l;
	uint8_t h;

	NAME_(addr_l)(cpssp);
	l = CHIP_(mrb)(cpssp, cpssp->NAME.addr + 0);
	h = CHIP_(mrb)(cpssp, cpssp->NAME.addr + 1);
	cpssp->NAME.addr = ((uint16_t) h << 8) | (uint16_t) l;
	cpssp->NAME.addr += cpssp->NAME.y;
}

static void
NAME_(addr_ind_hl)(struct cpssp *cpssp)
{
	uint8_t l;
	uint8_t h;

	NAME_(addr_hl)(cpssp);
	l = CHIP_(mrb)(cpssp, cpssp->NAME.addr + 0);
	h = CHIP_(mrb)(cpssp, cpssp->NAME.addr + 1);
	cpssp->NAME.addr = ((uint16_t) h << 8) | (uint16_t) l;
}

static void
NAME_(load_t1_akku)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = cpssp->NAME.akku;
}

static void
NAME_(load_t1_x)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = cpssp->NAME.x;
}

static void
NAME_(load_t1_y)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = cpssp->NAME.y;
}

static void
NAME_(load_t1_flags)(struct cpssp *cpssp)
{
	cpssp->NAME.t1
		= (cpssp->NAME.flag_n << 7)
		| (cpssp->NAME.flag_v << 6)
		| (1 << 5)
		| (cpssp->NAME.flag_b << 4)
		| (cpssp->NAME.flag_d << 3)
		| (cpssp->NAME.flag_i << 2)
		| (cpssp->NAME.flag_z << 1)
		| (cpssp->NAME.flag_c << 0);
}

static void
NAME_(load_t2_akku)(struct cpssp *cpssp)
{
	cpssp->NAME.t2 = cpssp->NAME.akku;
}

static void
NAME_(load_t2_x)(struct cpssp *cpssp)
{
	cpssp->NAME.t2 = cpssp->NAME.x;
}

static void
NAME_(load_t2_y)(struct cpssp *cpssp)
{
	cpssp->NAME.t2 = cpssp->NAME.y;
}

static void
NAME_(load_t2_sp)(struct cpssp *cpssp)
{
	cpssp->NAME.t2 = cpssp->NAME.sp;
}

static void
NAME_(load_t2_imm)(struct cpssp *cpssp)
{
	cpssp->NAME.t2 = NAME_(mfb)(cpssp);
}

static void
NAME_(load_t2_ind)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = CHIP_(mrb)(cpssp, cpssp->NAME.addr);
}

static void
NAME_(store_akku)(struct cpssp *cpssp)
{
	cpssp->NAME.akku = cpssp->NAME.t0;
}

static void
NAME_(store_x)(struct cpssp *cpssp)
{
	cpssp->NAME.x = cpssp->NAME.t0;
}

static void
NAME_(store_y)(struct cpssp *cpssp)
{
	cpssp->NAME.y = cpssp->NAME.t0;
}

static void
NAME_(store_sp)(struct cpssp *cpssp)
{
	cpssp->NAME.sp = cpssp->NAME.t0;
}

static void
NAME_(store_flags)(struct cpssp *cpssp)
{
	cpssp->NAME.flag_n = (((cpssp->NAME.t0) >> 7) & 1);
	cpssp->NAME.flag_v = (((cpssp->NAME.t0) >> 6) & 1);
	cpssp->NAME.flag_b = (((cpssp->NAME.t0) >> 4) & 1);
	cpssp->NAME.flag_d = (((cpssp->NAME.t0) >> 3) & 1);
	cpssp->NAME.flag_i = (((cpssp->NAME.t0) >> 2) & 1);
	cpssp->NAME.flag_z = (((cpssp->NAME.t0) >> 1) & 1);
	cpssp->NAME.flag_c = (((cpssp->NAME.t0) >> 0) & 1);
}

static void
NAME_(store_ind)(struct cpssp *cpssp)
{
	CHIP_(mwb)(cpssp, cpssp->NAME.addr, cpssp->NAME.t0);
}

static void
NAME_(do_bpl)(struct cpssp *cpssp)
{
	if (! cpssp->NAME.flag_n) {
		cpssp->NAME.pc += (int16_t) (int8_t) cpssp->NAME.t2;
	}
}

static void
NAME_(do_bmi)(struct cpssp *cpssp)
{
	if (cpssp->NAME.flag_n) {
		cpssp->NAME.pc += (int16_t) (int8_t) cpssp->NAME.t2;
	}
}

static void
NAME_(do_bvc)(struct cpssp *cpssp)
{
	if (! cpssp->NAME.flag_v) {
		cpssp->NAME.pc += (int16_t) (int8_t) cpssp->NAME.t2;
	}
}

static void
NAME_(do_bvs)(struct cpssp *cpssp)
{
	if (cpssp->NAME.flag_v) {
		cpssp->NAME.pc += (int16_t) (int8_t) cpssp->NAME.t2;
	}
}

static void
NAME_(do_bcc)(struct cpssp *cpssp)
{
	if (! cpssp->NAME.flag_c) {
		cpssp->NAME.pc += (int16_t) (int8_t) cpssp->NAME.t2;
	}
}

static void
NAME_(do_bcs)(struct cpssp *cpssp)
{
	if (cpssp->NAME.flag_c) {
		cpssp->NAME.pc += (int16_t) (int8_t) cpssp->NAME.t2;
	}
}

static void
NAME_(do_bne)(struct cpssp *cpssp)
{
	if (! cpssp->NAME.flag_z) {
		cpssp->NAME.pc += (int16_t) (int8_t) cpssp->NAME.t2;
	}
}

static void
NAME_(do_beq)(struct cpssp *cpssp)
{
	if (cpssp->NAME.flag_z) {
		cpssp->NAME.pc += (int16_t) (int8_t) cpssp->NAME.t2;
	}
}

static void
NAME_(do_mov)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t2;
}

static void
NAME_(do_mov_nz)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t2;
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_bit)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 & cpssp->NAME.t2;
	cpssp->NAME.flag_n = (cpssp->NAME.t2 >> 7) & 1;
	cpssp->NAME.flag_v = (cpssp->NAME.t2 >> 6) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_ora)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 | cpssp->NAME.t2;
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_and)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 & cpssp->NAME.t2;
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_eor)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 ^ cpssp->NAME.t2;
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_adc)(struct cpssp *cpssp)
{
	uint16_t res;

	if (cpssp->NAME.flag_d) {
		assert(0);
	} else {
		res = ((uint16_t) cpssp->NAME.t1 + (uint16_t) cpssp->NAME.t2 + cpssp->NAME.flag_c) & 0x1ff;
		cpssp->NAME.t0 = res & 0xff;
		cpssp->NAME.flag_c = (res >> 8) & 1;
	}
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_sbc)(struct cpssp *cpssp)
{
	uint16_t res;

	if (cpssp->NAME.flag_d) {
		assert(0);
	} else {
		res = ((uint16_t) cpssp->NAME.t1 + (uint16_t) ~cpssp->NAME.t2 + cpssp->NAME.flag_c) & 0x1ff;
		cpssp->NAME.t0 = res & 0xff;
		cpssp->NAME.flag_c = (res >> 8) & 1;
	}
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_alu)(struct cpssp *cpssp, uint8_t op)
{
	switch (op) {
	case 0:
		NAME_(do_ora)(cpssp);
		break;
	case 1:
		NAME_(do_and)(cpssp);
		break;
	case 2:
		NAME_(do_eor)(cpssp);
		break;
	case 3:
		NAME_(do_adc)(cpssp);
		break;
	case 6:
		NAME_(do_sbc)(cpssp);
		break;
	case 7:
		NAME_(do_sbc)(cpssp);
		break;
	}
}

static void
NAME_(do_asl)(struct cpssp *cpssp)
{
	uint16_t res;

	res = (uint16_t) cpssp->NAME.t2 << 1;
	cpssp->NAME.t0 = res & 0xff;
	cpssp->NAME.flag_c = (res >> 8) & 1;
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_rol)(struct cpssp *cpssp)
{
	uint16_t res;

	res = ((uint16_t) cpssp->NAME.t2 << 1) | cpssp->NAME.flag_c;
	cpssp->NAME.t0 = res & 0xff;
	cpssp->NAME.flag_c = (res >> 8) & 1;
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_lsr)(struct cpssp *cpssp)
{
	uint16_t res;

	res = cpssp->NAME.t2;
	cpssp->NAME.t0 = (res >> 1) & 0xff;
	cpssp->NAME.flag_c = res & 1;
	cpssp->NAME.flag_n = (cpssp->NAME.t0 >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_ror)(struct cpssp *cpssp)
{
	uint16_t res;

	res = (((uint16_t) cpssp->NAME.flag_c) << 8) | cpssp->NAME.t2;
	cpssp->NAME.t0 = ((res) >> 1) & 0xff;
	cpssp->NAME.flag_c = ((res)) & 1;
	cpssp->NAME.flag_n = ((cpssp->NAME.t0) >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_dec)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t2 - 1;
	cpssp->NAME.flag_n = ((cpssp->NAME.t0) >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_inc)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t2 + 1;
	cpssp->NAME.flag_n = ((cpssp->NAME.t0) >> 7) & 1;
	cpssp->NAME.flag_z = cpssp->NAME.t0 == 0;
}

static void
NAME_(do_shift)(struct cpssp *cpssp, uint8_t op)
{
	switch (op) {
	case 0:
		NAME_(do_asl)(cpssp);
		break;
	case 1:
		NAME_(do_rol)(cpssp);
		break;
	case 2:
		NAME_(do_lsr)(cpssp);
		break;
	case 3:
		NAME_(do_ror)(cpssp);
		break;
	case 6:
		NAME_(do_dec)(cpssp);
		break;
	case 7:
		NAME_(do_inc)(cpssp);
		break;
	}
}

static void
NAME_(do_jmp)(struct cpssp *cpssp)
{
	cpssp->NAME.pc = cpssp->NAME.addr;
}

static void
NAME_(do_push)(struct cpssp *cpssp)
{
	CHIP_(mwb)(cpssp, 0x0100 | cpssp->NAME.sp, cpssp->NAME.t1);
	cpssp->NAME.sp--;
}

static void
NAME_(do_pop)(struct cpssp *cpssp)
{
	cpssp->NAME.sp++;
	cpssp->NAME.t0 = CHIP_(mrb)(cpssp, 0x0100 | cpssp->NAME.sp);
}

static void
NAME_(step)(struct cpssp *cpssp)
{
	uint8_t inst;

	if (setjmp(cpssp->NAME.jmp_env)) goto end;

	if (cpssp->state_interrupt_pending) {
		cpssp->NAME.addr = 0xfffe;
		/* FIXME */

	} else {
		inst = NAME_(mfb)(cpssp);
		switch (inst) {
		case 0x00: /* BRK */
			cpssp->NAME.addr = 0xfffe;
			/* FIXME */
			break;
		case 0x20: /* JSR $hl */
			NAME_(addr_hl)(cpssp);
			cpssp->NAME.t1 = ((cpssp->NAME.pc) >> 0) & 0xff;
			NAME_(do_push)(cpssp);
			cpssp->NAME.t1 = ((cpssp->NAME.pc) >> 8) & 0xff;
			NAME_(do_push)(cpssp);
			cpssp->NAME.pc = cpssp->NAME.addr;
			break;
		case 0x40: /* RTI */
			NAME_(do_pop)(cpssp);
			NAME_(store_flags)(cpssp);
			NAME_(do_pop)(cpssp);
			cpssp->NAME.pc = ((uint16_t) cpssp->NAME.t0 << 8) | (cpssp->NAME.pc & 0xff);
			NAME_(do_pop)(cpssp);
			cpssp->NAME.pc = (cpssp->NAME.pc & 0xff00) | cpssp->NAME.t0;
			break;
		case 0x60: /* RTS */
			/* Not correct - FIXME */
			NAME_(do_pop)(cpssp);
			cpssp->NAME.pc = ((uint16_t) cpssp->NAME.t0 << 8) | (cpssp->NAME.pc & 0xff);
			NAME_(do_pop)(cpssp);
			cpssp->NAME.pc = (cpssp->NAME.pc & 0xff00) | cpssp->NAME.t0;
			break;
		/* case 0x80: Undefined */
		case 0xa0: /* LDY #$n */
			NAME_(load_t2_imm)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_y)(cpssp);
			break;
		case 0xc0: /* CPY #$n */
			NAME_(load_t1_y)(cpssp);
			NAME_(load_t2_imm)(cpssp);
			NAME_(do_sbc)(cpssp);
			break;
		case 0xe0: /* CPX #$n */
			NAME_(load_t1_x)(cpssp);
			NAME_(load_t2_imm)(cpssp);
			NAME_(do_sbc)(cpssp);
			break;

		case 0x01: /* ORA ($l,X) */
		case 0x21: /* AND ($l,X) */
		case 0x41: /* EOR ($l,X) */
		case 0x61: /* ADC ($l,X) */
		/* case 0x81: See below. */
		/* case 0xa1: See below. */
		case 0xc1: /* CMP ($l,X) */
		case 0xe1: /* SBC ($l,X) */
			NAME_(addr_ind_l_x)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_alu)(cpssp, (inst >> 5) & 7);
			if (inst != 0xc1) {
				NAME_(store_akku)(cpssp);
			}
			break;
		case 0x81: /* STA ($l,X) */
			NAME_(addr_ind_l_x)(cpssp);
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xa1: /* LDA ($l,X) */
			NAME_(addr_ind_l_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;

		/* case 0x02: Undefined */
		/* case 0x22: Undefined */
		/* case 0x42: Undefined */
		/* case 0x62: Undefined */
		/* case 0x82: Undefined */
		case 0xa2:
			NAME_(load_t2_imm)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_x)(cpssp);
			break;
		/* case 0xc2: Undefined */
		/* case 0xe2: Undefined */

		/* case 0x03: Undefined */
		/* case 0x23: Undefined */
		/* case 0x43: Undefined */
		/* case 0x63: Undefined */
		/* case 0x83: Undefined */
		/* case 0xa3: Undefined */
		/* case 0xc3: Undefined */
		/* case 0xe3: Undefined */

		/* case 0x04: Undefined */
		case 0x24: /* BIT $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_bit)(cpssp);
			break;
		/* case 0x44: Undefined */
		/* case 0x64: Undefined */
		case 0x84: /* STY $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t2_y)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xa4: /* LDY $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_y)(cpssp);
			break;
		case 0xc4: /* CPY $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t1_y)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_sbc)(cpssp);
			break;
		case 0xe4: /* CPX $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t1_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_sbc)(cpssp);
			break;

		case 0x05: /* ORA $l */
		case 0x25: /* AND $l */
		case 0x45: /* EOR $l */
		case 0x65: /* ADC $l */
		/* case 0x85: See below. */
		/* case 0xa5: See below. */
		case 0xc5: /* CMP $l */
		case 0xe5: /* SBC $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_alu)(cpssp, (inst >> 5) & 7);
			if (inst != 0xc5) {
				NAME_(store_akku)(cpssp);
			}
			break;
		case 0x85: /* STA $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xa5: /* LDA $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;

		case 0x06: /* ASL $l */
		case 0x26: /* ROL $l */
		case 0x46: /* LSR $l */
		case 0x66: /* ROR $l */
		/* case 0x86: See below. */
		/* case 0xa6: See below. */
		case 0xc6: /* DEC $l */
		case 0xe6: /* INC $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_shift)(cpssp, (inst >> 5) & 7);
			NAME_(store_ind)(cpssp);
			break;
		case 0x86: /* STX $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t2_x)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xa6: /* LDX $l */
			NAME_(addr_l)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_x)(cpssp);
			break;

		/* case 0x07: FIXME */
		/* case 0x27: FIXME */
		/* case 0x47: FIXME */
		/* case 0x67: FIXME */
		/* case 0x87: FIXME */
		/* case 0xa7: FIXME */
		/* case 0xc7: FIXME */
		/* case 0xe7: FIXME */

		case 0x08: /* PHP */
			NAME_(load_t1_flags)(cpssp);
			NAME_(do_push)(cpssp);
			break;
		case 0x28: /* PLP */
			NAME_(do_pop)(cpssp);
			NAME_(store_flags)(cpssp);
			break;
		case 0x48: /* PHA */
			NAME_(load_t1_akku)(cpssp);
			NAME_(do_push)(cpssp);
			break;
		case 0x68: /* PLA */
			NAME_(do_pop)(cpssp);
			NAME_(store_akku)(cpssp);
			break;
		case 0x88: /* DEY */
			NAME_(load_t1_y)(cpssp);
			NAME_(do_dec)(cpssp);
			NAME_(store_y)(cpssp);
			break;
		case 0xa8: /* TAY */
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_y)(cpssp);
			break;
		case 0xc8: /* INY */
			NAME_(load_t1_y)(cpssp);
			NAME_(do_inc)(cpssp);
			NAME_(store_y)(cpssp);
			break;
		case 0xe8: /* INX */
			NAME_(load_t1_x)(cpssp);
			NAME_(do_inc)(cpssp);
			NAME_(store_x)(cpssp);
			break;

		case 0x09: /* ORA #$n */
		case 0x29: /* AND #$n */
		case 0x49: /* EOR #$n */
		case 0x69: /* ADC #$n */
		/* case 0x89: See below. */
		/* case 0xa9: See below. */
		case 0xc9: /* CMP #$n */
		case 0xe9: /* SBC #$n */
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_imm)(cpssp);
			NAME_(do_alu)(cpssp, (inst >> 5) & 7);
			if (inst != 0xc9) {
				NAME_(store_akku)(cpssp);
			}
			break;
		/* case 0x89: Undefined */
		case 0xa9: /* LDA #$n */
			NAME_(load_t2_imm)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;

		case 0x0a: /* ASL */
		case 0x2a: /* ROL */
		case 0x4a: /* LSR */
		case 0x6a: /* ROR */
		/* case 0x8a: See below. */
		/* case 0xaa: See below. */
		case 0xca: /* DEX */
		/* case 0xea: Special? See below. */
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_shift)(cpssp, (inst >> 5) & 7);
			NAME_(store_akku)(cpssp);
			break;
		case 0x8a: /* TXA */
			NAME_(load_t2_x)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;
		case 0xaa: /* TAX */
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_x)(cpssp);
			break;
		case 0xea: /* NOP */
			break;

		/* case 0x0b: Undefined */
		/* case 0x2b: Undefined */
		/* case 0x4b: Undefined */
		/* case 0x6b: Undefined */
		/* case 0x8b: Undefined */
		/* case 0xab: Undefined */
		/* case 0xcb: Undefined */
		/* case 0xeb: Undefined */

		/* case 0x0c: Undefined. */
		case 0x2c: /* BIT $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_bit)(cpssp);
			break;
		case 0x4c: /* JMP $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(do_jmp)(cpssp);
			break;
		case 0x6c: /* JMP ($hl) */
			NAME_(addr_ind_hl)(cpssp);
			NAME_(do_jmp)(cpssp);
			break;
		case 0x8c: /* STY $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t2_y)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xac: /* LDY $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_y)(cpssp);
			break;
		case 0xcc: /* CPY $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t1_y)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_sbc)(cpssp);
			break;
		case 0xec: /* CPX $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t1_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_sbc)(cpssp);
			break;

		case 0x0d: /* ORA $hl */
		case 0x2d: /* AND $hl */
		case 0x4d: /* EOR $hl */
		case 0x6d: /* ADC $hl */
		/* case 0x8d: See below. */
		/* case 0xad: See below. */
		case 0xcd: /* CMP $hl */
		case 0xed: /* SBC $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_alu)(cpssp, (inst >> 5) & 7);
			if (inst != 0xcd) {
				NAME_(store_akku)(cpssp);
			}
			break;
		case 0x8d: /* STA $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xad: /* LDA $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;

		case 0x0e: /* ASL $hl */
		case 0x2e: /* ROL $hl */
		case 0x4e: /* LSR $hl */
		case 0x6e: /* ROR $hl */
		/* case 0x8e: See below. */
		/* case 0xae: See below. */
		case 0xce: /* DEC $hl */
		case 0xee: /* INC $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_shift)(cpssp, (inst >> 5) & 7);
			NAME_(store_ind)(cpssp);
			break;
		case 0x8e: /* STX $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t2_x)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xae: /* LDX $hl */
			NAME_(addr_hl)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_x)(cpssp);
			break;

		/* case 0x0f: Undefined */
		/* case 0x2f: Undefined */
		/* case 0x4f: Undefined */
		/* case 0x6f: Undefined */
		/* case 0x8f: Undefined */
		/* case 0xaf: Undefined */
		/* case 0xcf: Undefined */
		/* case 0xef: Undefined */

		case 0x10: /* BPL #$n */
		case 0x30: /* BMI #$n */
		case 0x50: /* BVC #$n */
		case 0x70: /* BVS #$n */
		case 0x90: /* BCC #$n */
		case 0xb0: /* BCS #$n */
		case 0xd0: /* BNE #$n */
		case 0xf0: /* BEQ #$n */
			NAME_(load_t2_imm)(cpssp);
			switch (inst) {
			case 0x10: NAME_(do_bpl)(cpssp); break;
			case 0x30: NAME_(do_bmi)(cpssp); break;
			case 0x50: NAME_(do_bvc)(cpssp); break;
			case 0x70: NAME_(do_bvs)(cpssp); break;
			case 0x90: NAME_(do_bcc)(cpssp); break;
			case 0xb0: NAME_(do_bcs)(cpssp); break;
			case 0xd0: NAME_(do_bne)(cpssp); break;
			case 0xf0: NAME_(do_beq)(cpssp); break;
			}
			break;

		case 0x11: /* ORA ($l),Y */
		case 0x31: /* AND ($l),Y */
		case 0x51: /* EOR ($l),Y */
		case 0x71: /* ADC ($l),Y */
		/* case 0x91: See below. */
		/* case 0xb1: See below. */
		case 0xd1: /* CMP ($l),Y */
		case 0xf1: /* SBC ($l),Y */
			NAME_(addr_ind_l_y)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_alu)(cpssp, (inst >> 5) & 7);
			if (inst != 0xd1) {
				NAME_(store_akku)(cpssp);
			}
			break;
		case 0x91: /* STA ($l),Y */
			NAME_(addr_ind_l_y)(cpssp);
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xb1: /* LDA ($l),Y */
			NAME_(addr_ind_l_y)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;

		/* case 0x12: Undefined */
		/* case 0x32: Undefined */
		/* case 0x52: Undefined */
		/* case 0x72: Undefined */
		/* case 0x92: Undefined */
		/* case 0xb2: Undefined */
		/* case 0xd2: Undefined */
		/* case 0xf2: Undefined */

		/* case 0x13: Undefined */
		/* case 0x33: Undefined */
		/* case 0x53: Undefined */
		/* case 0x73: Undefined */
		/* case 0x93: Undefined */
		/* case 0xb3: Undefined */
		/* case 0xd3: Undefined */
		/* case 0xf3: Undefined */

		/* case 0x14: Undefined */
		/* case 0x34: Undefined */
		/* case 0x54: Undefined */
		/* case 0x74: Undefined */
		case 0x94: /* STY $l,X */
			NAME_(addr_l_x)(cpssp);
			NAME_(load_t2_y)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xb4: /* LDY $l,X */
			NAME_(addr_l_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_y)(cpssp);
			break;
		/* case 0xd4: Undefined */
		/* case 0xf4: Undefined */

		case 0x15: /* ORA $l,X */
		case 0x35: /* AND $l,X */
		case 0x55: /* EOR $l,X */
		case 0x75: /* ADC $l,X */
		/* case 0x95: See below. */
		/* case 0xb5: See below. */
		case 0xd5: /* CMP $l,X */
		case 0xf5: /* SBC $l,X */
			NAME_(addr_l_x)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_alu)(cpssp, (inst >> 5) & 7);
			if (inst != 0xd5) {
				NAME_(store_akku)(cpssp);
			}
			break;
		case 0x95: /* STA $l,X */
			NAME_(addr_l_x)(cpssp);
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xb5: /* LDA $l,X */
			NAME_(addr_l_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;

		case 0x16: /* ASL $l,X */
		case 0x36: /* ROL $l,X */
		case 0x56: /* LSR $l,X */
		case 0x76: /* ROR $l,X */
		/* case 0x96: See below. */
		/* case 0xb6: See below. */
		case 0xd6: /* DEC $l,X */
		case 0xf6: /* INC $l,X */
			NAME_(addr_l_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_shift)(cpssp, (inst >> 5) & 7);
			NAME_(store_ind)(cpssp);
			break;
		case 0x96: /* STX $l,Y */
			NAME_(addr_l_y)(cpssp);
			NAME_(load_t2_x)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xb6: /* LDX $l,Y */
			NAME_(addr_l_y)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_x)(cpssp);
			break;

		/* case 0x17: Undefined */
		/* case 0x37: Undefined */
		/* case 0x57: Undefined */
		/* case 0x77: Undefined */
		/* case 0x97: Undefined */
		/* case 0xb7: Undefined */
		/* case 0xd7: Undefined */
		/* case 0xf7: Undefined */

		case 0x18: /* CLC */
			cpssp->NAME.flag_c = 0;
			break;
		case 0x38: /* SEC */
			cpssp->NAME.flag_c = 1;
			break;
		case 0x58: /* CLI */
			cpssp->NAME.flag_i = 0;
			break;
		case 0x78: /* SEI */
			cpssp->NAME.flag_i = 1;
			break;
		case 0x98: /* TYA */
			NAME_(load_t2_y)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;
		case 0xb8: /* CLV */
			cpssp->NAME.flag_v = 0;
			break;
		case 0xd8: /* CLD */
			cpssp->NAME.flag_d = 0;
			break;
		case 0xf8: /* SED */
			cpssp->NAME.flag_d = 1;
			break;

		case 0x19: /* ORA $hl,Y */
		case 0x39: /* ORA $hl,Y */
		case 0x59: /* ORA $hl,Y */
		case 0x79: /* ORA $hl,Y */
		/* case 0x99: See below. */
		/* case 0xb9: See below. */
		case 0xd9: /* ORA $hl,Y */
		case 0xf9: /* ORA $hl,Y */
			NAME_(addr_hl_y)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_alu)(cpssp, (inst >> 5) & 7);
			if (inst != 0xd9) {
				NAME_(store_akku)(cpssp);
			}
			break;
		case 0x99: /* STA $hl,Y */
			NAME_(addr_hl_y)(cpssp);
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xb9: /* LDA $hl,Y */
			NAME_(addr_hl_y)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;

		/* case 0x1a: Undefined */
		/* case 0x3a: Undefined */
		/* case 0x5a: Undefined */
		/* case 0x7a: Undefined */
		case 0x9a: /* TXS */
			NAME_(load_t2_x)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_sp)(cpssp);
			break;
		case 0xba: /* TSX */
			NAME_(load_t2_sp)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_x)(cpssp);
			break;

		/* case 0xda: Undefined */
		/* case 0xfa: Undefined */

		/* case 0x1b: Undefined */
		/* case 0x3b: Undefined */
		/* case 0x5b: Undefined */
		/* case 0x7b: Undefined */
		/* case 0x9b: Undefined */
		/* case 0xbb: Undefined */
		/* case 0xdb: Undefined */
		/* case 0xfb: Undefined */

		/* case 0x1c: Undefined */
		/* case 0x3c: Undefined */
		/* case 0x5c: Undefined */
		/* case 0x7c: Undefined */
		/* case 0x9c: Undefined */
		case 0xbc: /* LDY $hl,X */
			NAME_(addr_hl_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_y)(cpssp);
			break;
		/* case 0xdc: Undefined */
		/* case 0xfc: Undefined */

		case 0x1d: /* ORA $hl,X */
		case 0x3d: /* AND $hl,X */
		case 0x5d: /* EOR $hl,X */
		case 0x7d: /* ADC $hl,X */
		/* case 0x9d: See below. */
		/* case 0xbd: See below. */
		case 0xdd: /* CMP $hl,X */
		case 0xfd: /* SBC $hl,X */
			NAME_(addr_hl_x)(cpssp);
			NAME_(load_t1_akku)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_alu)(cpssp, (inst >> 5) & 7);
			if (inst != 0xdd) {
				NAME_(store_akku)(cpssp);
			}
			break;
		case 0x9d: /* STA $hl,X */
			NAME_(addr_hl_x)(cpssp);
			NAME_(load_t2_akku)(cpssp);
			NAME_(do_mov)(cpssp);
			NAME_(store_ind)(cpssp);
			break;
		case 0xbd: /* LDA $hl,X */
			NAME_(addr_hl_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_akku)(cpssp);
			break;

		case 0x1e: /* ASL $hl,X */
		case 0x3e: /* ROL $hl,X */
		case 0x5e: /* LSR $hl,X */
		case 0x7e: /* ROR $hl,X */
		/* case 0x9e: Special? See below. */
		/* case 0xbe: See below. */
		case 0xde: /* DEC $hl,X */
		case 0xfe: /* INC $hl,X */
			NAME_(addr_hl_x)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_shift)(cpssp, (inst >> 5) & 7);
			NAME_(store_ind)(cpssp);
			break;
		/* case 0x9e: Undefined */
		case 0xbe: /* LDX $hl,Y */
			NAME_(addr_hl_y)(cpssp);
			NAME_(load_t2_ind)(cpssp);
			NAME_(do_mov_nz)(cpssp);
			NAME_(store_x)(cpssp);
			break;

		/* case 0x1f: FIXME */
		/* case 0x3f: FIXME */
		/* case 0x5f: FIXME */
		/* case 0x7f: FIXME */
		/* case 0x9f: FIXME */
		/* case 0xbf: FIXME */
		/* case 0xdf: FIXME */
		/* case 0xff: FIXME */

		default:
			assert(0);
		}
	}
end:	;
}

#endif /* BEHAVIOR */
