#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2008 Rob Shortt <rob@tvcentric.com>
# Copyright 2015-2022 Univention GmbH
# Author: Rob Shortt <rob@tvcentric.com>

"""
This is a notifier implementation using Twisted - http://www.twistedmatrix.com/
Twisted is an async framework that has much in common with pynotifier and kaa.

Here are some links of interest to aid development and debugging:

The reactor base class, posixbase, and selectreactor:
http://twistedmatrix.com/trac/browser/trunk/twisted/internet/base.py
http://twistedmatrix.com/trac/browser/trunk/twisted/internet/posixbase.py
http://twistedmatrix.com/trac/browser/trunk/twisted/internet/selectreactor.py

Timers and scheduling:
http://twistedmatrix.com/projects/core/documentation/howto/time.html

Twisted doc index:
http://twistedmatrix.com/projects/core/documentation/howto/index.html
"""

from __future__ import absolute_import

from typing import Dict  # noqa: F401

from twisted.internet import reactor, task
from twisted.internet.interfaces import IReadDescriptor, IWriteDescriptor
#from zope.interface import implements
from zope.interface import implementer

from . import _DispathCB, _FileLike, _get_fd, _SocketCB, _TimerCB, dispatch, log  # noqa: F401

IO_READ = 1
IO_WRITE = 2
IO_EXCEPT = 4

__sockobjs = {
    IO_READ: {},
    IO_WRITE: {},
}  # type: Dict[int, Dict]
__timers = {}  # type: Dict
_TimerID = int
__timer_id = 0
__dispatch_timer = None


@implementer(IReadDescriptor)
class SocketReadCB:
    """
    An object to implement Twisted's IReadDescriptor.  When there is data
    available on the socket doRead() will get called.
    """
    #implements(IReadDescriptor)

    def __init__(self, socket, method):
        # type: (_FileLike, _SocketCB) -> None
        self.socket = socket
        self.method = method

    def doRead(self):
        # type: () -> None
        """
        Call the callback method with the socket as the only argument.  If it
        returns False remove this socket from our notifier.
        """
        if not self.method(self.socket):
            socket_remove(self.socket, IO_READ)

    def fileno(self):
        # type: () -> int
        return _get_fd(self.socket)

    def logPrefix(self):
        # type: () -> str
        return "notifier"

    def connectionLost(self, reason):
        # type: (object) -> None
        # Should we do more?
        log.error("connection lost on socket fd=%s" % self.fileno())


@implementer(IWriteDescriptor)
class SocketWriteCB:
    """
    An object to implement Twisted's IWriteDescriptor.  When there is data
    available on the socket doWrite() will get called.
    """
    #implements(IWriteDescriptor)

    def __init__(self, socket, method):
        # type: (_FileLike, _SocketCB) -> None
        self.socket = socket
        self.method = method

    def doWrite(self):
        # type: () -> None
        """
        Call the callback method with the socket as the only argument.  If it
        returns False remove this socket from our notifier.
        """
        if not self.method(self.socket):
            socket_remove(self.socket, IO_WRITE)

    def fileno(self):
        # type: () -> int
        return _get_fd(self.socket)

    def logPrefix(self):
        # type: () -> str
        return "notifier"

    def connectionLost(self, reason):
        # type: (object) -> None
        # Should we do more?
        log.error("connection lost on socket fd=%s" % self.fileno())


def socket_add(id, method, condition=IO_READ):
    # type: (_FileLike, _SocketCB, int) -> None
    """
    The first argument specifies a socket, the second argument has to be a
    function that is called whenever there is data ready in the socket.

    Objects that implement Twisted's IRead/WriteDescriptor interfaces get
    passed to the reactor to monitor.
    """
    if condition == IO_READ:
        sr = SocketReadCB(id, method)
        reactor.addReader(sr)
        __sockobjs[condition][id] = sr
    elif condition == IO_WRITE:
        sw = SocketWriteCB(id, method)
        reactor.addWriter(sw)
        __sockobjs[condition][id] = sw


def socket_remove(id, condition=IO_READ):
    # type: (_FileLike, int) -> None
    """
    Removes the IRead/WriteDescriptor object with this socket from
    the Twisted reactor.
    """
    sockobj = __sockobjs[condition].get(id)

    if sockobj:
        if condition == IO_READ:
            reactor.removeReader(sockobj)
        elif condition == IO_WRITE:
            reactor.removeWriter(sockobj)
        del __sockobjs[condition][id]


def timer_add(interval, method):
    # type: (int, _TimerCB) -> _TimerID
    """
    The first argument specifies an interval in milliseconds, the second
    argument a function. This function is called after interval
    seconds. If it returns true it's called again after interval
    seconds, otherwise it is removed from the scheduler. The third
    (optional) argument is a parameter given to the called
    function. This function returns an unique identifier which can be
    used to remove this timer
    """
    global __timer_id

    try:
        __timer_id += 1
    except OverflowError:
        __timer_id = 0

    def _method(id_):
        # type: (_TimerID) -> None
        if not method():
            timer_remove(id_)

    t = task.LoopingCall(_method, __timer_id)
    t.start(interval / 1000.0, now=False)
    __timers[__timer_id] = t

    return __timer_id


def timer_remove(id):
    # type: (_TimerID) -> None
    """
    Removes the timer identified by the unique ID from the main loop.
    """
    t = __timers.get(id)
    if t is not None:
        t.stop()
        del __timers[id]


def dispatcher_add(method):
    # type: (_DispathCB) -> None
    dispatch.dispatcher_add(method)


dispatcher_remove = dispatch.dispatcher_remove


def step(sleep=True, external=True):
    # type: (bool, bool) -> None
    if reactor.running:
        try:
            t = sleep and reactor.running and reactor.timeout()
            if dispatch.dispatcher_count():
                t = dispatch.MIN_TIMER / 1000.0
            reactor.doIteration(t)
            reactor.runUntilCurrent()
        except Exception:
            log.error("problem running reactor - exiting")
            raise SystemExit
        if external:
            dispatch.dispatcher_run()
    else:
        log.info("reactor stopped - exiting")
        raise SystemExit


def loop():
    # type: () -> None
    """
    Instead of calling reactor.run() here we must call step() so we get a
    chance to call dispatch.dispatcher_run().  Otherwise the dispatchers
    would have to be run in a timer, making something like the 'step' signal
    not getting called every iteration of the main loop like it was intended.

    We could also decide between reactor.run() and step() and if we use step()
    just setup a Timer for the dispatchers at a reasonable rate, i.e.:

            global __dispatch_timer
            __dispatch_timer = task.LoopingCall(dispatch.dispatcher_run)
            __dispatch_timer.start(dispatch.MIN_TIMER/1000.0) # 10x / second
            # or
            # __dispatch_timer.start(1.0/30) # 30x / second
    """
    while True:
        try:
            step()
        except (SystemExit, KeyboardInterrupt):
            log.debug("exiting loop")
            break
        except Exception as exc:
            log.debug("exception: %s", exc, exc_info=True)
            raise


def _init():
    # type: () -> None
    reactor.startRunning(installSignalHandlers=True)
