#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2006, 2011, 2012 Andreas Büsching <crunchy@bitkipper.net>
# Copyright 2015-2022 Univention GmbH
# Author: Andreas Büsching	<crunchy@bitkipper.net>

"""
simple interface to handle threads synchron to the notifier loop
"""

import functools
import sys
import threading
import traceback
from typing import Any, Callable, Generic, List, Optional, Tuple, Type, TypeVar, Union  # noqa: F401

import notifier

__all__ = ['Simple']

_threads = []  # type: List[Simple]

_T = TypeVar("_T")


class Simple(Generic[_T]):
	"""A simple class to start a thread and getting notified when the
	thread is finished. Meaning this class helps to handle threads that
	are meant for doing some calculations and returning the
	result. Threads that need to communicate with the main thread can
	not be handled by this class.

	If an exception is raised during the execution of the thread that is
	based on BaseException it is caught and returned as the result of
	the thread.

	Arguments:
	name: a name that might be used to identify the thread. It is not required to be unique.
	function: the main function of the thread
	callback: function that is invoked when the thread is dead. This function gets two arguments:
	thread: this thread object
	result: return value of the thread function.
	"""

	def __init__(self, name, function, callback):
		# type: (str, Callable[..., _T], Callable[[Simple, Union[BaseException, None, _T]], None]) -> None
		self._name = name
		self._function = function
		self._callback = callback
		self._result = None  # type: Union[BaseException, _T, None]
		self._trace = None  # type: Optional[List[str]]
		self._exc_info = None  # type: Optional[Tuple[Optional[Type[BaseException]], Optional[BaseException], None]]
		self._finished = False
		self._thread = threading.Thread(target=self._run, name=self._name)
		self._lock = threading.Lock()
		if not _threads:
			notifier.dispatcher_add(_simple_threads_dispatcher)
		_threads.append(self)

	def run(self):
		# type: () -> None
		"""Starts the thread"""
		self._thread.start()

	def _run(self):
		# type: () -> None
		"""Encapsulates the given thread function to handle the return
		value in a thread-safe way and to catch exceptions raised from
		within it."""
		try:
			result = self._function()  # type: Union[BaseException, _T]
			trace = None  # type: Optional[List[str]]
			exc_info = None  # type: Optional[Tuple[Optional[Type[BaseException]], Optional[BaseException], None]]
		except BaseException as exc:
			try:
				etype, value, tb = sys.exc_info()
				trace = traceback.format_tb(tb)
				exc_info = (etype, value, None)
			finally:
				etype = value = tb = None
			result = exc
		self.lock()
		try:
			self._result = result
			self._trace = trace
			self._exc_info = exc_info
			self._finished = True
		finally:
			self.unlock()

	@property
	def result(self):
		# type: () -> Union[BaseException, _T, None]
		"""Contains the result of the thread function or the exception
		that occurred during thread processing"""
		return self._result

	@property
	def trace(self):
		# type: () -> Optional[List[str]]
		"""Contains a formatted traceback of the occurred exception during
		thread processing. If no exception has been raised the value is None"""
		return self._trace

	@property
	def exc_info(self):
		# type: () -> Optional[Tuple[Optional[Type[BaseException]], Optional[BaseException], None]]
		"""Contains information about the exception that has occurred
		during the execution of the thread. The value is the some as
		returned by sys.exc_info(). If no exception has been raised the
		value is None"""
		return self._exc_info

	@property
	def name(self):
		# type: () -> str
		return self._name

	@property
	def finished(self):
		# type: () -> bool
		"""If the thread is finished the property contains the value
		True else False."""
		return self._finished

	def lock(self):
		# type: () -> None
		"""Locks a thread local lock object"""
		self._lock.acquire()

	def unlock(self):
		# type: () -> None
		"""Unlocks a thread local lock object"""
		self._lock.release()

	def announce(self):
		# type: () -> None
		self._callback(self, self._result)


class Enhanced(Simple):
	def __init__(self, function, callback):
		# type: (Callable, Callable[[Simple, object], None]) -> None
		Simple.__init__(self, '__enhanced__', function, callback)
		self._signals = []  # type: List[Tuple[str, Tuple[object, ...]]]

	def signal_emit(self, signal, *args):
		# type: (str, *object) -> None
		self.lock()
		self._signals.append((signal, args))
		self.unlock()


def _simple_threads_dispatcher():
	# type: () -> bool
	"""Dispatcher function checking for finished threads"""
	for task in _threads[:]:
		task.lock()
		try:
			if task.finished:
				task.announce()
				_threads.remove(task)
			elif isinstance(task, Enhanced):
				for signal, args in task._signals:
					task.signal_emit(signal, *args)
				task._signals = []
		finally:
			task.unlock()

	return bool(_threads)


def threaded(finished_func):
	# type: (Callable[..., Any]) -> Any
	"""A decorator function making it simple to start a thread. Just
	add the decorator to the function that should be the main thread
	function. The argument is the function that should be invoked when
	the thread has finished"""

	def inner_thread(func):
		# type: (Callable[..., Any]) -> Callable[..., Any]
		def wrapped(*args, **kwargs):
			# type: (*Any, **Any) -> None
			thread = Enhanced(notifier.Callback(func, *args, **kwargs), finished_func)
			thread.run()
		return functools.wraps(func)(wrapped)
	return inner_thread
