#ifndef _RHEO_BRANCH_H
#define _RHEO_BRANCH_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/field.h"
namespace rheolef { 

class __obranch;
class __iobranch;
class __branch_header;
class __const_branch_header;
class __const_branch_finalize;

/*Class:branch
@cindex animation
@cindex continuation methods
@cindex time-dependent problems
@clindex field

NAME:  @code{branch} - (t,uh(t)) finite element function
DESCRIPTION:       
  Stores a @code{field} together with its associated parameter
  value: a @code{branch} variable represents a pair (t,uh(t)) for
  a specific value of the parameter t.
  Applications concern time-dependent problems and continuation methods.

  This class is convenient for file inputs/outputs
  and building graphical animations.

EXAMPLES:
  Coming soon...

LIMITATIONS:
  This class is under development.

  The @code{branch} class store pointers on @code{field} class
  without reference counting. Thus, @code{branch} automatic 
  variables cannot be returned by functions. @code{branch}
  variable are limited to local variables.

  At each step, meshes are reloaded and spaces are rebuilted, even
  when they are identical.

  The @code{vtk} render does not support mesh change during
  the loop (e.g. when using adaptive mesh process).

AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France.
    @url{mailto:Pierre.Saramito@imag.fr}
END:
*/

// =============================================================================
// class definition
// =============================================================================

//<branch:  
class branch : public std::vector<std::pair<std::string,field> > {
public :

// allocators:

	branch ();
	branch (const std::string& parameter_name, const std::string& u_name);
	branch (const std::string& parameter_name, const std::string& u_name, const std::string& p_name);
	~branch ();

// accessors:

	const Float&  parameter () const;
	const std::string& parameter_name () const;
	size_type     n_value () const;
	size_type     n_field () const;

// modifiers:

	void set_parameter (const Float& value);

// input/output:

	// get/set current value
	friend std::istream& operator >> (std::istream&, branch&);
	friend std::ostream& operator << (std::ostream&, const branch&);

	__branch_header         header ();
	__const_branch_header   header () const;
	__const_branch_finalize finalize () const;

	__obranch operator() (const Float& t, const field& u);
	__obranch operator() (const Float& t, const field& u, const field& p);

	__iobranch operator() (Float& t, field& u);
	__iobranch operator() (Float& t, field& u, field& p);
//>branch:

// implementation

public:
	void put_header (std::ostream&) const;
	void get_header (std::istream&);
	void put_finalize (std::ostream&) const;

	friend class __obranch;
	friend class __iobranch;

// file formats
	void put_header_rheolef (std::ostream&) const;
	void put_event_rheolef (std::ostream&) const;

	void put_header_gnuplot    (std::ostream&) const;
	void put_event_gnuplot     (std::ostream&) const;
	void put_finalize_gnuplot  (std::ostream&) const;

	void put_header_vtk    (std::ostream&) const;
	void put_geometry_vtk  (std::ostream&) const;
	void put_event_vtk     (std::ostream&) const;
	void put_finalize_vtk  (std::ostream&) const;

	void put_header_paraview    (std::ostream&) const;
	void put_event_paraview     (std::ostream&) const;
	void put_finalize_paraview  (std::ostream&) const;
// data:
   protected:
	std::string           _parameter_name;
    	Float                 _parameter_value;
    	size_type             _n_value;
    	mutable size_type     _count_value;
    	mutable std::ostream* _p_data_out;
    	mutable bool          _header_printed;
};
std::istream& operator >> (std::istream&, branch&);
std::ostream& operator << (std::ostream&, const branch&);
// =============================================================================
// inlined
// =============================================================================
inline
branch::branch ()
 : std::vector<std::pair<std::string,field> >(0),
   _parameter_name ("unnamed"),
   _parameter_value(std::numeric_limits<Float>::max()),
   _n_value        (std::numeric_limits<size_type>::max()),
   _count_value    (std::numeric_limits<size_type>::max()),
   _p_data_out	   (0),
   _header_printed (false)
{
}
inline
branch::branch (const std::string& parameter_name, const std::string& field_name)
 : std::vector<std::pair<std::string,field> >(1),
   _parameter_name (parameter_name),
   _parameter_value(std::numeric_limits<Float>::max()),
   _n_value        (std::numeric_limits<size_type>::max()),
   _count_value    (std::numeric_limits<size_type>::max()),
   _p_data_out	   (0),
   _header_printed (false)
{
    check_macro(parameter_name.length() != 0, "empty parameter name not allowed");
    check_macro(field_name.length() != 0, "empty field name not allowed");
    operator[](0).first = field_name;
}
inline
branch::branch (const std::string& parameter_name, const std::string& u_name, const std::string& p_name)
 : std::vector<std::pair<std::string,field> >(2),
   _parameter_name (parameter_name),
   _parameter_value(std::numeric_limits<Float>::max()),
   _n_value        (std::numeric_limits<size_type>::max()),
   _count_value    (std::numeric_limits<size_type>::max()),
   _p_data_out	   (0),
   _header_printed (false)
{
    check_macro(parameter_name.length() != 0, "empty parameter name not allowed");
    check_macro(u_name.length() != 0, "empty field #1 name not allowed");
    check_macro(p_name.length() != 0, "empty field #2 name not allowed");
    operator[](0).first = u_name;
    operator[](1).first = p_name;
}
// -----------------------------------------------------------------------------
// accessors
// -----------------------------------------------------------------------------
inline
const Float&
branch::parameter () const
{
    return _parameter_value;
}
inline
const std::string&
branch::parameter_name () const
{
    return _parameter_name;
}
inline
branch::size_type
branch::n_value () const
{
    return _n_value;
}
inline
branch::size_type
branch::n_field () const
{
    return size();
}
// -----------------------------------------------------------------------------
// modifiers
// -----------------------------------------------------------------------------
inline
void 
branch::set_parameter (const Float& x)
{
    _parameter_value = x;
}
// -----------------------------------------------------------------------------
// io header wrapper
// -----------------------------------------------------------------------------
class __branch_header {
    public:
	__branch_header (branch& b) : _b(b) {}
	friend std::istream& operator >> (std::istream& in, __branch_header h) { 
		h._b.get_header(in); return in; }
	friend std::ostream& operator << (std::ostream& out, __branch_header h) { 
		h._b.put_header(out); return out; }
    protected:
    	branch& _b;
	friend class __const_branch_header;
};
class __const_branch_header {
    public:
	__const_branch_header (const branch& b) : _b(b) {}
	__const_branch_header (__branch_header h) : _b(h._b) {}
	friend std::ostream& operator << (std::ostream& out, __const_branch_header h) { 
		h._b.put_header(out); return out; }
    protected:
    	const branch& _b;
};
inline
__branch_header
branch::header ()
{
    return *this;
}
inline
__const_branch_header
branch::header () const
{
    return *this;
}
// -----------------------------------------------------------------------------
// o finalize wrapper
// -----------------------------------------------------------------------------
class __const_branch_finalize {
    public:
	__const_branch_finalize (const branch& b) : _b(b) {}
	friend std::ostream& operator << (std::ostream& out, __const_branch_finalize h) { 
		h._b.put_finalize(out); return out; }
    protected:
    	const branch& _b;
};
inline
__const_branch_finalize
branch::finalize () const
{
    return *this;
}
// -----------------------------------------------------------------------------
// io value wrapper
// -----------------------------------------------------------------------------
class __obranch {
    public:
        __obranch (std::ostream& (*put)(std::ostream&, const branch&), const branch& x)
         : _put(put), _x(x) {}
        friend std::ostream& operator << (std::ostream& os, __obranch m)
         { m._put (os, m._x); return os; }
    private:
	std::ostream&    (*_put) (std::ostream&, const branch&);
        const branch& _x;
};
class __iobranch {
    public:
        __iobranch (std::ostream& (*put)(std::ostream&, const branch&), 
	            std::istream& (*get)(std::istream&, branch&),
		    branch& x, 
		    Float*  pt = 0, 
		    field*  pu = 0, 
		    field*  pp = 0)
         : _put(put), _get(get), _px(&x), _pt(pt), _pu(pu), _pp(pp) {}
	friend std::ostream& operator << (std::ostream& os, __iobranch m);
	friend std::istream& operator >> (std::istream& is, __iobranch m);
    private:
	std::ostream&    (*_put) (std::ostream&, const branch&);
	std::istream&    (*_get) (std::istream&, branch&);
        branch *_px;
        Float  *_pt;
        field  *_pu;
        field  *_pp;
};
inline
std::ostream& operator << (std::ostream& os, __iobranch m) {
    m._put (os, *m._px);
    return os;
}
inline
std::istream& operator >> (std::istream& is, __iobranch m) { 
    m._get (is, *m._px); 
    if (m._pt) { *m._pt = (*m._px).parameter(); }
    if (m._pu) { *m._pu = (*m._px)[0].second; }
    if (m._pp) { *m._pp = (*m._px)[1].second; }
    return is;
}
inline
__obranch
branch::operator() (const Float& t, const field& u)
{
    check_macro (size() >= 1, "attempt to output a 1-field branch when a " 
	<< size() << "-field one was supplied");
    _parameter_value = t;
    operator[](0).second = u;
    return __obranch (operator<<, *this);
}
inline
__iobranch
branch::operator() (Float& t, field& u)
{
    check_macro (size() >= 1, "attempt to input/output a 1-field branch when a " 
	<< size() << "-field one was supplied");
    _parameter_value = t;
    operator[](0).second = u;
    return __iobranch (operator<<, operator>>, *this, &t, &u);
}
inline
__obranch
branch::operator() (const Float& t, const field& u, const field& p)
{
    check_macro (size() >= 2, "attempt to output a 2-field branch when a " 
	<< size() << "-field one was supplied");
    _parameter_value = t;
    operator[](0).second = u;
    operator[](1).second = p;
    return __obranch (operator<<, *this);
}
inline
__iobranch
branch::operator() (Float& t, field& u, field& p)
{
    check_macro (size() >= 2, "attempt to output a 2-field branch when a " 
	<< size() << "-field one was supplied");
    _parameter_value = t;
    operator[](0).second = u;
    operator[](1).second = p;
    return __iobranch (operator<<, operator>>, *this, &t, &u, &p);
}
}// namespace rheolef
#endif // define_RHEO_BRANCH_H
