#!/usr/bin/python3
"""Simple HTTP Proxy for ucs-test."""
# Inspired by <http://effbot.org/librarybook/simplehttpserver.htm>

import base64
import os
import shutil
from optparse import OptionParser

from six.moves import BaseHTTPServer, http_client
from six.moves.urllib.error import HTTPError
from six.moves.urllib.parse import unquote, urlsplit, urlunsplit
from six.moves.urllib.request import Request, urlopen

PORT = 3128


class Proxy(BaseHTTPServer.BaseHTTPRequestHandler):
	server_version = "UCSTestProxy/1.0"

	def do_GET(self):
		self.common(data=True)

	def do_HEAD(self):
		self.common(data=False)

	def common(self, data=True):
		if options.authorization:
			try:
				auth = self.headers.get('Proxy-Authorization', '')
				if not auth.lower().startswith('basic '):
					raise KeyError("Only Basic authentication: %s" % auth)
				auth = auth[len('Basic '):]
				auth = base64.b64decode(auth).decode('ISO8859-1')
				username, password = auth.split(':', 1)
				username, password = unquote(username), unquote(password)
				if username != options.username:
					msg = f"Username: {username} != {options.username}"
					if options.verbose:
						self.log_error(msg)
					raise KeyError(msg)
				if password != options.password:
					msg = f"Password: {password} != {options.password}"
					if options.verbose:
						self.log_error(msg)
					raise KeyError(msg)
			except KeyError as exc:
				self.send_response(http_client.PROXY_AUTHENTICATION_REQUIRED)
				self.send_header('WWW-Authenticate', f'Basic realm="{options.realm}"')
				self.send_header('Content-type', 'text/html; charset=UTF-8')
				self.end_headers()
				self.wfile.write((f'<html><body><h1>Error: Proxy authorization needed</h1>{exc}</body></html>').encode('UTF-8'))
				return
		# rewrite url
		url = urlsplit(self.path)
		u = list(url)
		# The proxy gets a verbatim copy of the URL, which might contain the
		# target site credentials. urllib doesn't handle this, strip it.
		if '@' in u[1]:
			u[1] = u[1].split('@', 1)[1]
		# Fake DNS resolve of configured hostname to localhost
		if options.translate:
			if url.hostname == options.translate:
				u[1] = u[1].replace(options.translate, 'localhost')
		url = urlunsplit(u)
		try:
			req = Request(url=url, headers=self.headers)
			if options.verbose:
				for k, v in self.headers.items():
					self.log_message(f"> {k}: {v}")
			fp = urlopen(req)
		except HTTPError as exc:
			fp = exc
			if options.verbose:
				self.log_error("%d %s" % (fp.code, fp.msg))

		self.send_response(fp.code)
		via = f'1.0 {httpd.server_name}'
		for k, v in fp.headers.items():
			if k.lower() == 'via':
				via = f"{via}, {v}"
			elif k.lower() in ('server', 'date'):  # Std-Hrds by BaseHTTPReqHand
				continue
			elif k.lower() == 'transfer-encoding':
				continue
			else:
				if options.verbose:
					self.log_message(f"< {k}: {v}")
				self.send_header(k, v)
		self.send_header('Via', via)
		self.end_headers()
		if data:
			shutil.copyfileobj(fp, self.wfile)
		fp.close()


if __name__ == '__main__':
	parser = OptionParser()
	parser.add_option('-p', '--port', action='store', dest='port', type='int', default=PORT, help='TCP port number')
	parser.add_option('-a', '--authorization', action='store_true', dest='authorization', default=False, help='Require authorization')
	parser.add_option('-u', '--username', action='store', dest='username', default='username', help='User name for HTTP Proxy authorization, unquoted')
	parser.add_option('-w', '--password', action='store', dest='password', default='password', help='Password for HTTP Proxy authorization, unquoted')
	parser.add_option('-r', '--realm', action='store', dest='realm', default='realm', help='Realm for HTTP Proxy authorization')
	parser.add_option('-t', '--translate', action='store', dest='translate', metavar='HOSTNAME', help='Translate requests for this host name to localhost')
	parser.add_option('-f', '--fork', action='store_true', dest='fork', default=False, help='Fork daemon process')
	parser.add_option('-v', '--verbose', action='store_true', dest='verbose', default=False, help='Output verbose informations')
	(options, arguments) = parser.parse_args()

	httpd = BaseHTTPServer.HTTPServer(('', int(options.port)), Proxy)
	if options.fork:
		pid = os.fork()
		if pid == 0:
			for fd in range(3):
				os.close(fd)
				fd2 = os.open(os.devnull, fd == 0 and os.O_RDONLY or os.O_WRONLY)
				if fd2 != fd:
					os.dup2(fd2, fd)
					os.close(fd2)
			httpd.serve_forever()
		else:
			print("proxy_pid=%d proxy_port=%d" % (pid, httpd.server_port))
	else:
		try:
			print("proxy_pid=%d proxy_port=%d" % (os.getpid(), httpd.server_port))
			httpd.serve_forever()
		except KeyboardInterrupt:
			pass
