#
# Like what you see? Join us!
# https://www.univention.com/about-us/careers/vacancies/
#
# SPDX-FileCopyrightText: 2004-2025 Univention GmbH
# SPDX-License-Identifier: AGPL-3.0-only

"""|UDM| module for |DNS| reverse zones"""

from typing import Any  # noqa: F401

import ldap.dn

import univention.admin.filter
import univention.admin.handlers
import univention.admin.localization
from univention.admin import configRegistry
from univention.admin.handlers.dns import (
    ARPA_IP4, ARPA_IP6, escapeSOAemail, is_dns, is_reverse_zone, is_zone, unescapeSOAemail,
)
from univention.admin.layout import Group, Tab


translation = univention.admin.localization.translation('univention.admin.handlers.dns')
_ = translation.translate

module = 'dns/reverse_zone'
operations = ['add', 'edit', 'remove', 'search']
columns = ['nameserver']
childs = True
childmodules = ['dns/ptr_record', 'dns/ns_record']
short_description = _('DNS: Reverse lookup zone')
object_name = _('Reverse lookup zone')
object_name_plural = _('Reverse lookup zones')
long_description = _('Map IP addresses back to hostnames.')
options = {
    'default': univention.admin.option(
        short_description=short_description,
        default=True,
        objectClasses=['top', 'dNSZone'],
    ),
}
property_descriptions = {
    'subnet': univention.admin.property(
        short_description=_('Subnet'),
        long_description=_('The networks address in forward notation.'),
        syntax=univention.admin.syntax.reverseLookupSubnet,
        include_in_default_search=True,
        required=True,
        may_change=False,
        identifies=True,
    ),
    'zonettl': univention.admin.property(
        short_description=_('Zone time to live'),
        long_description=_('The time this entry may be cached.'),
        syntax=univention.admin.syntax.UNIX_TimeInterval,
        required=True,
        default=(('3', 'hours'), []),
        dontsearch=True,
    ),
    'contact': univention.admin.property(
        short_description=_('Contact person'),
        long_description=_('The email address of the person responsible for this zone.'),
        syntax=univention.admin.syntax.emailAddressThatMayEndWithADot,
        required=True,
        default=('root@%(domainname)s.' % configRegistry, []),
    ),
    'serial': univention.admin.property(
        short_description=_('Serial number'),
        long_description=_('The sequence number for this zone. Updates automatically.'),
        syntax=univention.admin.syntax.integer,
        required=True,
        default=('1', []),
    ),
    'refresh': univention.admin.property(
        short_description=_('Refresh interval'),
        long_description=_('The time interval secondary DNS servers use to check the zone for updates.'),
        syntax=univention.admin.syntax.UNIX_TimeInterval,
        required=True,
        default=(('8', 'hours'), []),
    ),
    'retry': univention.admin.property(
        short_description=_('Retry interval'),
        long_description=_('The time interval secondary DNS servers use to retry failed refresh updates.'),
        syntax=univention.admin.syntax.UNIX_TimeInterval,
        required=True,
        default=(('2', 'hours'), []),
    ),
    'expire': univention.admin.property(
        short_description=_('Expiry interval'),
        long_description=_('The time interval after which secondary DNS servers will expire failed zones.'),
        syntax=univention.admin.syntax.UNIX_TimeInterval,
        required=True,
        default=(('7', 'days'), []),
    ),
    'ttl': univention.admin.property(
        short_description=_('Negative time to live'),
        long_description=_('The time interval "not found" answers are cached.'),
        syntax=univention.admin.syntax.UNIX_TimeInterval,
        required=True,
        default=(('1', 'days'), []),
    ),
    'nameserver': univention.admin.property(
        short_description=_('Name server'),
        long_description=_('The FQDNs of the servers serving this zone. A FQDN must end with a dot.'),
        syntax=univention.admin.syntax.dnsHostname,
        multivalue=True,
        required=True,
    ),
}

layout = [
    Tab(_('General'), _('Basic settings'), layout=[
        Group(_('General reverse lookup zone settings'), layout=[
            'subnet',
            'zonettl',
            'nameserver',
        ]),
    ]),
    Tab(_('Start of authority'), _('Primary name server information'), layout=[
        Group(_('Start of authority'), layout=[
            'contact',
            'serial',
            ['refresh', 'retry'],
            ['expire', 'ttl'],
        ]),
    ]),
]


def mapSubnet(subnet, encoding=()):
    """
    Map subnet to reverse zone.
    >>> mapSubnet('0123:4567:89ab:cdef')
    'f.e.d.c.b.a.9.8.7.6.5.4.3.2.1.0.ip6.arpa'
    >>> mapSubnet('0123:4567:89ab:cd')
    'd.c.b.a.9.8.7.6.5.4.3.2.1.0.ip6.arpa'
    >>> mapSubnet('1.2.3')
    '3.2.1.in-addr.arpa'
    """
    if ':' in subnet:  # IPv6
        subnet = '%s%s' % ('.'.join(subnet.replace(':', '')[::-1]), ARPA_IP6)
    else:
        subnet = '%s%s' % ('.'.join(subnet.split('.')[::-1]), ARPA_IP4)
    return subnet.encode(*encoding)


def unmapSubnet(zone, encoding=()):
    """
    Map reverse zone to subnet.
    >>> unmapSubnet([b'f.e.d.c.b.a.9.8.7.6.5.4.3.2.1.0.ip6.arpa'])
    '0123:4567:89ab:cdef'
    >>> unmapSubnet([b'd.c.b.a.9.8.7.6.5.4.3.2.1.0.ip6.arpa'])
    '0123:4567:89ab:cd'
    >>> unmapSubnet([b'3.2.1.in-addr.arpa'])
    '1.2.3'
    """
    if isinstance(zone, list):
        zone = zone[0]
    zone = zone.decode(*encoding)
    if zone.endswith(ARPA_IP6):  # IPv6
        zone = zone[:-len(ARPA_IP6)]
        zone = zone.split('.')[::-1]
        return ':'.join([''.join(zone[i:i + 4]) for i in range(0, len(zone), 4)])
    elif zone.endswith(ARPA_IP4):  # IPv4
        zone = zone[:-len(ARPA_IP4)]
        q = zone.split('.')
        q.reverse()
        return '.'.join(q)
    else:
        raise ValueError('Neither an IPv4 nor an IPv6 reverse address')


mapping = univention.admin.mapping.mapping()
mapping.register('subnet', 'zoneName', mapSubnet, unmapSubnet, encoding='ASCII')
mapping.register('zonettl', 'dNSTTL', univention.admin.mapping.mapUNIX_TimeInterval, univention.admin.mapping.unmapUNIX_TimeInterval)
mapping.register('nameserver', 'nSRecord', encoding='ASCII')


class object(univention.admin.handlers.simpleLdap):
    module = module

    def __init__(
        self,
        co,  # type: None
        lo,  # type: univention.admin.uldap.access
        position,  # type: univention.admin.uldap.position | None
        dn='',  # type: str
        superordinate=None,  # type: univention.admin.handlers.simpleLdap | None
        attributes=None,  # type: univention.admin.handlers._Attributes | None
    ):  # type: (...) -> None
        univention.admin.handlers.simpleLdap.__init__(self, co, lo, position, dn, superordinate, attributes=attributes)
        if not self.dn and not self.position:
            raise univention.admin.uexceptions.insufficientInformation(_('Neither DN nor position given.'))

    def open(self):
        # type: () -> None
        univention.admin.handlers.simpleLdap.open(self)

        soa = self.oldattr.get('sOARecord', [b''])[0].split(b' ')
        if len(soa) > 6:
            self['contact'] = unescapeSOAemail(soa[1].decode('ASCII'))
            self['serial'] = soa[2].decode('ASCII')
            self['refresh'] = univention.admin.mapping.unmapUNIX_TimeInterval(soa[3])
            self['retry'] = univention.admin.mapping.unmapUNIX_TimeInterval(soa[4])
            self['expire'] = univention.admin.mapping.unmapUNIX_TimeInterval(soa[5])
            self['ttl'] = univention.admin.mapping.unmapUNIX_TimeInterval(soa[6])

        self.save()

    def _ldap_modlist(self):
        # type: () -> list[tuple[str, Any, Any]]
        ml = univention.admin.handlers.simpleLdap._ldap_modlist(self)
        if self.hasChanged(['nameserver', 'contact', 'serial', 'refresh', 'retry', 'expire', 'ttl']):
            if self['contact'] and not self['contact'].endswith('.'):
                self['contact'] += '.'
            for i, ns in enumerate(self['nameserver']):
                if ns and ':' not in ns and '.' in ns and not ns.endswith('.'):
                    self['nameserver'][i] += '.'
            refresh = univention.admin.mapping.mapUNIX_TimeInterval(self['refresh'])
            retry = univention.admin.mapping.mapUNIX_TimeInterval(self['retry'])
            expire = univention.admin.mapping.mapUNIX_TimeInterval(self['expire'])
            ttl = univention.admin.mapping.mapUNIX_TimeInterval(self['ttl'])
            soa = b'%s %s %s %s %s %s %s' % (self['nameserver'][0].encode('ASCII'), escapeSOAemail(self['contact']).encode('ASCII'), self['serial'].encode('ASCII'), refresh, retry, expire, ttl)
            ml.append(('sOARecord', self.oldattr.get('sOARecord', []), soa))
        return ml

    def _ldap_pre_modify(self):
        # type: () -> None
        super()._ldap_pre_modify()
        # update SOA record
        if not self.hasChanged('serial'):
            self['serial'] = str(int(self['serial']) + 1)

    def _ldap_addlist(self):
        # type: () -> list[tuple[str, Any]]
        return [*super()._ldap_addlist(), ('relativeDomainName', [b'@'])]

    # FIXME: there should be general solution; subnet is just a naming
    # attribute (though calculated from rdn)
    def description(self):
        # type: () -> str
        if self.exists():
            rdn_value = ldap.dn.str2dn(self.dn)[0][0][1].encode('UTF-8')
            return unmapSubnet([rdn_value])
        return self['subnet']

    @classmethod
    def unmapped_lookup_filter(cls):
        # type: () -> univention.admin.filter.conjunction
        return univention.admin.filter.conjunction('&', [
            univention.admin.filter.expression('objectClass', 'dNSZone'),
            univention.admin.filter.expression('sOARecord', '*', escape=False),
            univention.admin.filter.conjunction('|', [
                univention.admin.filter.expression('zoneName', '*%s' % ARPA_IP4, escape=False),
                univention.admin.filter.expression('zoneName', '*%s' % ARPA_IP6, escape=False),
            ]),
        ])


lookup = object.lookup
lookup_filter = object.lookup_filter


def identify(dn, attr):
    # type: (str, univention.admin.handlers._Attributes) -> bool
    return bool(
        is_zone(attr)
        and is_dns(attr)
        and is_reverse_zone(attr),
    )
