#!/usr/bin/python3
# SPDX-FileCopyrightText: 2016-2026 Univention GmbH
# SPDX-License-Identifier: AGPL-3.0-only

import socket
import subprocess

import ldap

import univention.admin.modules as udm_modules
import univention.admin.uldap
import univention.lib.misc
import univention.uldap
from univention.config_registry import handler_set as ucr_set, handler_unset as ucr_unset, ucr_live as configRegistry
from univention.lib.i18n import Translation
from univention.management.console.modules.diagnostic import MODULE, Critical, Instance, ProblemFixed


_ = Translation('univention-management-console-module-diagnostic').translate
run_descr = ["Trying to authenticate with machine password against LDAP  Similar to running: univention-ldapsearch -LLLs base dn"]
title = _('Check machine password')
description = _('Authentication with machine password against LDAP successful.')
links = [{
    'name': 'sdb',
    'href': 'https://help.univention.com/t/manually-trigger-server-password-change/6376',
    'label': _('Univention Support Database - Manually trigger server password change'),
}]


def fix_machine_password(umc_instance: Instance) -> None:
    role = configRegistry.get('server/role')
    valid_roles = ('domaincontroller_master', 'domaincontroller_backup', 'domaincontroller_slave', 'memberserver')
    if role in valid_roles:
        restore_machine_password(role, umc_instance.get_user_ldap_connection())

        if configRegistry.is_true('server/password/change', True):
            change_server_password()
        return run(umc_instance, retest=True)

    error_description = _('Unable to fix machine password on {}').format(role)
    raise Critical(description=error_description)


def reset_password_change(umc_instance: Instance) -> None:
    MODULE.process('Resetting server/password/change')
    ucr_unset(['server/password/change'])
    return run(umc_instance, retest=True)


def reset_password_interval(umc_instance: Instance) -> None:
    MODULE.process('Resetting server/password/interval=21')
    ucr_set(['server/password/interval=21'])
    return run(umc_instance, retest=True)


actions = {
    'fix_machine_password': fix_machine_password,
    'reset_password_change': reset_password_change,
    'reset_password_interval': reset_password_interval,
}


def check_machine_password(master: bool = True) -> bool:
    try:
        univention.uldap.getMachineConnection(ldap_master=master)
    except ldap.INVALID_CREDENTIALS:
        return False
    return True


def change_server_password() -> None:
    interval = configRegistry.get_int('server/password/interval', 21)
    ucr_set('server/password/interval=-1')
    try:
        cmd = ['/usr/lib/univention-server/server_password_change']
        output = subprocess.check_output(cmd, stderr=subprocess.STDOUT)
        MODULE.process('Output of server_password_change: %s', output.decode('UTF-8', 'replace'))
    except subprocess.CalledProcessError as exc:
        MODULE.error('Error running server_password_change')
        MODULE.error('Output: %s', exc.output.decode('UTF-8', 'replace'))
        error_descriptions = [
            _('Calling /usr/lib/univention-server/server_password_change failed.'),
            _('Please see {sdb} for more information.'),
        ]
        MODULE.error(' '.join(error_descriptions))
        raise Critical(description=' '.join(error_descriptions))
    finally:
        ucr_set(f'server/password/interval={interval}')


def restore_machine_password(role: str, ldap_connection: univention.admin.uldap.access) -> None:
    with open('/etc/machine.secret') as fob:
        password = fob.read().rstrip('\n')

    if not password:
        password = univention.lib.misc.createMachinePassword()
        with open('/etc/machine.secret', 'w') as fob:
            fob.write(password)

    computers = udm_modules.get(f'computers/{role}')
    position = univention.admin.uldap.position(ldap_connection.base)
    udm_modules.init(ldap_connection, position, computers)
    filter_expr = ldap.filter.filter_format('(cn=%s)', (socket.gethostname(),))
    for computer in computers.lookup(None, ldap_connection, filter_expr):
        MODULE.process('Restoring password of UDM computer object')
        computer.open()
        computer['password'] = password
        computer.modify()


def run(_umc_instance: Instance, retest: bool = False) -> None:
    error_descriptions = []
    buttons = [{
        'action': 'fix_machine_password',
        'label': _('Fix machine password'),
    }]

    is_master = configRegistry.get('server/role') == 'domaincontroller_master'
    if not is_master and not check_machine_password(master=False):
        error = _('Authentication against the local LDAP failed with the machine password.')
        error_descriptions.append(error)

    if not check_machine_password(master=True):
        error = _('Authentication against the Primary LDAP failed with the machine password.')
        error_descriptions.append(error)

    password_change = configRegistry.is_true('server/password/change', True)
    change_interval = configRegistry.get_int('server/password/interval', 21)

    error_change = _('Note that password rotation is disabled via the UCR variable server/password/change.')
    error_interval = _('Note that server/password/interval is set to {}.')

    if error_descriptions:
        note_sdb = _('See {sdb} for information on manual server password change.')
        error_descriptions.append(note_sdb)

        if not password_change:
            error_descriptions.append(error_change)
            buttons.append({
                'action': 'reset_password_change',
                'label': _('Set server/password/change=True'),
            })
        if change_interval < 1:
            error_descriptions.append(error_interval.format(change_interval))
            buttons.append({
                'action': 'reset_password_interval',
                'label': _('Set server/password/interval=21'),
            })

            MODULE.error('\n'.join(error_descriptions))
            raise Critical(description=' '.join(error_descriptions), buttons=buttons)
    if retest:
        raise ProblemFixed(buttons=[])


if __name__ == '__main__':
    from univention.management.console.modules.diagnostic import main
    main()
