/*
 * NodeTextureBackground.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

// setColor(), drawQuad(), drawSky() and drawGround() based on 
// render_Background() in Bindable.c of FreeWRL (ported by wu qingwei)
/*******************************************************************
 Copyright (C) 1998 Tuomas J. Lukka
 Copyright (C) 2002 John Stewart, CRC Canada.
 DISTRIBUTED WITH NO WARRANTY, EXPRESS OR IMPLIED.
 See the GNU Library General Public License (file COPYING in the distribution)
 for conditions of use and redistribution.
*********************************************************************/

#include <stdio.h>
#include "stdafx.h"

#include "NodeTextureBackground.h"
#include "Proto.h"
#include "FieldValue.h"
#include "MFColor.h"
#include "MFFloat.h"
#include "MFString.h"
#include "SFColor.h"
#include "SFRotation.h"
#include "NodeImageTexture.h"
#include "NodeViewpoint.h"
#include "Scene.h"
#include "Util.h"
#include "Field.h" // for FF_URL
#include "Node.h"

ProtoTextureBackground::ProtoTextureBackground(Scene *scene)
  : Proto(scene, "TextureBackground")
{
    addEventIn(SFBOOL, "set_bind");

    groundAngle.set(
          addExposedField(MFFLOAT, "groundAngle", new MFFloat(), 
                          new SFFloat(0.0f), new SFFloat(M_PI / 2.0f)));
    groundColor.set(
         addExposedField(MFCOLOR, "groundColor", new MFColor()));

    backTexture.set(
         addExposedField(SFNODE, "backTexture", new SFNode()));
    rightTexture.set(
         addExposedField(SFNODE, "rightTexture", new SFNode()));
    frontTexture.set(
         addExposedField(SFNODE, "frontTexture", new SFNode()));
    leftTexture.set(
         addExposedField(SFNODE, "leftTexture", new SFNode()));
    topTexture.set(
         addExposedField(SFNODE, "topTexture", new SFNode()));
    bottomTexture.set(
         addExposedField(SFNODE, "bottomTexture", new SFNode()));

    skyAngle.set(
         addExposedField(MFFLOAT, "skyAngle", new MFFloat(), new SFFloat(0.0f),
                         new SFFloat(M_PI)));

    float *colors = new float[3];
    colors[0] = colors[1] = colors[2] = 0.0f;
    skyColor.set(
          addExposedField(MFCOLOR, "skyColor", new MFColor(colors, 3)));
    addEventOut(SFBOOL, "isBound");
}

Node *
ProtoTextureBackground::create(Scene *scene)
{ 
    return new NodeTextureBackground(scene, this); 
}

NodeTextureBackground::NodeTextureBackground(Scene *scene, Proto *def)
  : Node(scene, def)
{
    for (int i = 0; i < 6; i++) {
        imageTextures[i] = (NodeImageTexture *) scene->createNode("ImageTexture");
        imageTextures[i]->ref();
    }
}

NodeTextureBackground::~NodeTextureBackground()
{
    for (int i = 0; i < 6; i++) {
        imageTextures[i]->unref();
    }
}

void
NodeTextureBackground::setField(int field, FieldValue *value)
{
    Node::setField(field, value);
}

void
NodeTextureBackground::apply()
{
    glPushAttrib(GL_ENABLE_BIT|GL_TEXTURE_BIT);
    glPushMatrix();

    glDisable(GL_DEPTH_TEST);
    glDisable(GL_LIGHTING);
    glDisable(GL_FOG);
    glDisable(GL_CULL_FACE);
    glLoadIdentity();
    const float *r = _scene->getCamera()->orientation()->getValue();
    glRotatef(-RAD2DEG(r[3]), r[0], r[1], r[2]);
    drawTextureBackgroundSphere();
    drawTextureBackgroundTextures();

    glPopMatrix();
    glPopAttrib();
}

void
NodeTextureBackground::setColor(const float *newColor)
{
    Util::myGlMaterialfv(GL_FRONT, GL_EMISSION, newColor);
    Util::myGlColor3f(newColor[0], newColor[1], newColor[2]);
}

void
NodeTextureBackground::drawQuad(float r, float va1, float va2, float h1, float h2, 
                         const float *newColor)
{
    float ha1 = h1 * 2.0 * M_PI;
    float ha2 = h2 * 2.0 * M_PI;
    glVertex3d(r * sin(va2) * cos(ha1), r * cos(va2), r * sin(va2) * sin(ha1));
    glVertex3d(r * sin(va2) * cos(ha2), r * cos(va2), r * sin(va2) * sin(ha2));
    if (newColor)
        setColor(newColor);
    glVertex3d(r * sin(va1) * cos(ha2), r * cos(va1), r * sin(va1) * sin(ha2));
    glVertex3d(r * sin(va1) * cos(ha1), r * cos(va1), r * sin(va1) * sin(ha1));
}

#define NUMBER_HORIZONTAL_POLYGONS 20

void
NodeTextureBackground::drawSky()
{
    float sc = TheApp->GetFarClippingPlaneDist() - 2;
    int hdiv = NUMBER_HORIZONTAL_POLYGONS;
    glBegin(GL_QUADS);
    if (skyColor()->getSFSize() == 1) {
        float va1 = 0;
        float va2 = M_PI / 2.0; 
        setColor(skyColor()->getValue(0));
        for (int v = 0; v < 2; v++) {
            for (float h = 0; h < hdiv; h++)
                drawQuad(sc, va1, va2, h / hdiv, (h + 1) / hdiv, NULL);
            va1 = va2;
            va2 = M_PI;
        }
    } else {
        float va1 = 0;
        float va2 = 0;
 
        if (skyAngle()->getSFSize() != 0) {
            va2 = skyAngle()->getValue(0);
            const float black[] = { 0, 0, 0 };
            const float* colors2 = black;

            for(int v = 0; v < skyColor()->getSFSize() - 1; v++) {
                if (v >= skyAngle()->getSFSize())
                    break;
                va2 = skyAngle()->getValue(v);
                const float* colors1 = skyColor()->getValue(v);
                colors2 = skyColor()->getValue(v + 1);

                for (float h = 0; h < hdiv; h++) {
                    setColor(colors2);
                    drawQuad(sc, va1, va2, h / hdiv, (h + 1) / hdiv, colors1);
                }
                va1 = va2;
            }

            if (va2 < M_PI) {
                setColor(colors2);
                for (float h = 0; h < hdiv; h++)
                    drawQuad(sc, M_PI, va2, h / hdiv, (h + 1) / hdiv, NULL);
            }
        }
    }
    glEnd();
}

void
NodeTextureBackground::drawGround()
{
    int hdiv = NUMBER_HORIZONTAL_POLYGONS;
    // where to put the ground quads 
    float sc = TheApp->GetFarClippingPlaneDist() - 4; 

    glBegin(GL_QUADS);
    if (groundColor()->getSFSize() == 1) {
        const float* gcolors1=groundColor()->getValue(0);
        setColor(gcolors1);
        for (float h = 0; h < hdiv; h++)
            drawQuad(sc, M_PI / 2.0, M_PI , h / hdiv, (h + 1) / hdiv, NULL);
    } else {
        float va1 = M_PI;
        for (int v = 0; v < groundColor()->getSFSize() - 1; v++) {
            const float* gcolors1 = groundColor()->getValue(v);
            const float* gcolors2 = groundColor()->getValue(v+1);

            float va2;
            if (v >= groundAngle()->getSFSize())
                break;
            va2 = M_PI - groundAngle()->getValue(v);

            for (float h = 0; h < hdiv; h++) {
                setColor(gcolors1);
                drawQuad(sc, va2, va1 , h / hdiv, (h + 1) / hdiv, gcolors2);
            }
            va1 = va2;
        }
    }
    glEnd();
}

void
NodeTextureBackground::drawTextureBackgroundSphere()
{
    glShadeModel(GL_SMOOTH);
    glDisable(GL_LIGHTING);

    if (skyColor())
        if (skyColor()->getSFSize() > 0)
            drawSky();

    if (groundColor())
        if (groundColor()->getSFSize()>0)
            drawGround();
}

void
NodeTextureBackground::drawTextureBackgroundTextures()
{
    glEnable(GL_LIGHTING);
    static float white[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    Util::myGlMaterialfv(GL_FRONT, GL_EMISSION, white);

    glScalef(40.0f, 40.0f, 40.0f);

    for (int i = 0; i < 6; i++) {
        if (imageTextures[i]->url()->getSize() != 0) {
            imageTextures[i]->bind();
            glBegin(GL_POLYGON);
            if (i < 4)
                for (int j = 0; j < 4; j++) {
                    glTexCoord2fv( boxTexCoords[j] );
                    glVertex3fv( boxCorners[boxIndices[i*4+j]] );
                }
            else
                for (int j = 0; j < 4; j++) {
                    glTexCoord2fv( boxTexCoords[(j+2)%4] );
                    glVertex3fv( boxCorners[boxIndices[i*4+j]] );
                }
            glEnd();
            imageTextures[i]->unbind();
        }
    }
}

void
NodeTextureBackground::preDraw()
{
//    _scene->addTextureBackground(this);
}

int
NodeTextureBackground::countPolygons(void)
{
    int ret = 0;
    if (skyColor())
        ret += NUMBER_HORIZONTAL_POLYGONS * skyColor()->getSFSize();
    if (groundColor())
        ret += NUMBER_HORIZONTAL_POLYGONS * groundColor()->getSFSize();
    for (int i = 0; i < 6; i++)
        if (imageTextures[i]->url()->getSize() > 0)
            ret++;
    return ret;
}

